<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model;

use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Model\Quote\Address;
use Magento\Quote\Model\Quote\AddressFactory;
use Magento\Sales\Model\Order;
use Novuna\Pbf\Api\Data\DraftOrderDataInterface;
use Novuna\Pbf\Model\DraftOrder\PaymentUrlGenerator;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteIdManager;
use Novuna\Pbf\Model\ResourceModel\DraftOrder as DraftOrderResource;

class DraftOrder extends AbstractModel implements DraftOrderDataInterface
{
    private CartRepositoryInterface $cartRepository;
    private QuoteIdManager $quoteIdManager;
    private PaymentUrlGenerator $paymentUrlGenerator;
    private ?CartInterface $quote = null;
    private AddressFactory $addressFactory;
    private ?Address $address = null;
    private Order $order;

    public function __construct(
        QuoteIdManager $quoteIdManager,
        PaymentUrlGenerator $paymentUrlGenerator,
        CartRepositoryInterface $cartRepository,
        Context $context,
        Registry $registry,
        AddressFactory $addressFactory,
        AbstractResource $resource = null,
        AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
        $this->cartRepository = $cartRepository;
        $this->quoteIdManager = $quoteIdManager;
        $this->paymentUrlGenerator = $paymentUrlGenerator;
        $this->addressFactory = $addressFactory;
    }

    protected function _construct()
    {
        $this->_init(DraftOrderResource::class);
    }

    public function getId(): ?int
    {
        $id = $this->getData(self::ID);
        return $id ? (int)$id : null;
    }

    public function setId($id)
    {
        return $this->setData(self::ID, $id);
    }

    public function getOrderId(): ?int
    {
        return $this->getData(self::ORDER_ID) ? (int)$this->getData(self::ORDER_ID) : null;
    }

    public function getName(): ?string
    {
        if ($this->getData('name')) {
            return $this->getData('name');
        }

        if ($this->getData(self::ID)) {
            return (string)$this->getData(self::ID);
        }

        return null;
    }

    public function setName($draftName)
    {
        return $this->setData('name', $draftName);
    }

    public function getReservedOrderId(): ?string
    {
        return $this->getQuote()->getReservedOrderId();
    }

    public function setOrderId($orderId)
    {
        return $this->setData(self::ORDER_ID, $orderId);
    }

    public function getQuoteId(): ?int
    {
        return (int)$this->getData(self::QUOTE_ID);
    }

    public function setQuoteId($quoteId)
    {
        return $this->setData(self::QUOTE_ID, $quoteId);
    }

    public function getEmail(): ?string
    {
        return $this->getShippingAddress()->getEmail();
    }

    public function getShippingTotal(): ?float
    {
        return floatval($this->getShippingAddress()->getShippingInclTax());
    }

    public function getSubTotalInclTax(): ?float
    {
        return floatval($this->getShippingAddress()->getSubtotalInclTax());
    }

    public function getFinanceAmount(): ?float
    {
        return floatval($this->getData(self::FINANCE_AMOUNT));
    }

    public function getDepositCaptureType(): int
    {
        return (int)$this->getData(self::PBF_TAKING_DEPOSIT);
    }

    public function getTotal(): ?float
    {
        return floatval($this->getShippingAddress()->getGrandTotal());
    }

    public function getPayDepositUrl(): string
    {
        return $this->paymentUrlGenerator->generate((string)$this->getQuote()->getId());
    }

    public function getAddress(): \Magento\Quote\Api\Data\AddressInterface
    {
        if (!$this->address) {
            $address = json_decode($this->getData(self::ADDRESS) ?? "", true) ?? [];
            if (empty($address)) {
                $this->address = $this->getQuote()->getShippingAddress()->setId(null)->setCustomerAddressId(null);
            } else {
                $this->address = $this->addressFactory->create()->setData($address);
            }
        }

        return $this->address;
    }

    public function setAddress(\Magento\Quote\Api\Data\AddressInterface $address)
    {
        $this->address = $address;
        $this->setData(self::ADDRESS, json_encode($address->getData()));
    }

    public function getShippingAddress(): \Magento\Quote\Api\Data\AddressInterface
    {
        return $this->getAddress();
    }

    public function getBillingAddress(): \Magento\Quote\Api\Data\AddressInterface
    {
        return $this->getAddress();
    }

    /**
     * @return \Magento\Quote\Api\Data\CartItemInterface[]
     */
    public function getLineItems(): ?array
    {
        return $this->getQuote()->getAllVisibleItems();
    }

    public function getPrivateId(): string
    {
        return (string)$this->getQuote()->getId();
    }

    public function getPublicId(): string
    {
        return $this->quoteIdManager->getMaskedById((int)$this->getQuote()->getId());
    }

    public function getQuote(): CartInterface
    {
        if ($this->quote === null) {
            $this->quote = $this->cartRepository->get($this->getQuoteId());
        }
        return $this->quote;
    }

    public function setQuote(CartInterface $quote)
    {
        $this->quote = $quote;
    }

    public function setOrder(Order $order)
    {
        $this->order = $order;
        $this->setOrderId($order->getId());
    }

    public function getOrder()
    {
        return $this->order;
    }

    public function draftArray(): array
    {
        return [
            'publicId' => $this->getPublicId(),
            'privateId' => $this->getPrivateId(),
            'paymentUrl' => $this->getPayDepositUrl()
        ];
    }
}
