<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Model\Widget;

use Laminas\Uri\Uri;
use Magento\Framework\App\Area;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\DataObject;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Integration\Api\IntegrationServiceInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Store\Model\StoresConfig;
use Novuna\Pbf\Model\StoreConfigProvider;

class PbfWidgetConfigProvider extends DataObject
{
    const XML_PATH_PAYBYFINANCE_CAN_APPLY_FROM_CART = 'pbf/general/can_aply_from_cart';
    const APPSERVER_DEV_URL = 'https://pbf-dev-magento.staging-pbffinancecalculator.info';
    const APPSERVER_URL = 'https://pbf-live-magento.pbffinancecalculator.info';
    const PROXIED_PATH = '/pbfwidget/'; //we will proxy all requests from that URL base path to appserver
    const PROXY_DESTINATION_PATH = '/magento/'; // destination base url part on appserver
    const INSTALLATION_WEBSOCKET_URL = '/magento/installWS';
    const THANK_YOU_JS_PATH = '/magento/static/thankyou.js';
    const CM3_VERSION = 3;
    const CM2_VERSION = 2;
    private $allowedScriptDomains = [
        'staging-pbffinancecalculator.info',
        'pbffinancecalculator.info',
        'dev-site.com'
    ];

    private StoreConfigProvider $storeConfigProvider;
    private Uri $uri;
    private StoreManagerInterface $storeManager;
    private SerializerInterface $serializer;
    private StoresConfig $storesConfig;
    /**
     * @var array|bool|float|int|string|null
     */
    private $widgetSettings;

    private RequestInterface $request;
    private ScopeConfigInterface $scopeConfig;
    private \Magento\Framework\App\State $appState;
    private IntegrationServiceInterface $integrationService;

    public function __construct(
        Uri $uri,
        StoreManagerInterface $storeManager,
        StoreConfigProvider $storeConfigProvider,
        SerializerInterface $serializer,
        RequestInterface $request,
        ScopeConfigInterface $scopeConfig,
        StoresConfig $storesConfig,
        \Magento\Framework\App\State $appState,
        IntegrationServiceInterface $integrationService
    ) {
        $this->storeConfigProvider = $storeConfigProvider;
        $this->uri = $uri;
        $this->storeManager = $storeManager;
        $this->serializer = $serializer;
        $this->request = $request;
        $this->scopeConfig = $scopeConfig;
        $this->storesConfig = $storesConfig;
        $this->appState = $appState;
        $this->integrationService = $integrationService;
    }

    /**
     * Get app server url depending on selected mode
     * @return string
     */
    public function getAppServerUrl(): string
    {

        if ($this->storeConfigProvider->getPbfMode() == StoreConfigProvider::MODE_TEST) {
            return self::APPSERVER_DEV_URL;
        }
        return self::APPSERVER_URL;
    }

    /**
     * check /V1/pbf/scriptTag for creating script tags
     * @param string $eventName
     * @return string|null
     */
    public function getScriptSrc(string $eventName = ''): ?string
    {
        $scriptSrc = $this->storeConfigProvider->getPbfWidgetScriptSrc($eventName, ScopeInterface::SCOPE_STORE,
            $this->storeManager->getStore()->getId());
        if ($scriptSrc) {
            $parsedUri = $this->uri->parse($scriptSrc);
            $scriptHost = $parsedUri->getHost();
            //@todo improve code
            $scriptHost = implode('.', array_slice(explode('.', $scriptHost), -2));
            return in_array($scriptHost, $this->allowedScriptDomains) ? $scriptSrc : '';
        }

        return '';
    }

    public function canApplyFromCart()
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_PAYBYFINANCE_CAN_APPLY_FROM_CART);
    }

    public function getDefaultStoreShopDomainHost()
    {
        $path = 'web/unsecure/base_url';

        $storeId = $this->storeManager->getDefaultStoreView()->getId();
        $url = $this->scopeConfig->getValue($path, ScopeInterface::SCOPE_STORE, $storeId);
        return parse_url($url, PHP_URL_HOST);
    }

    public function getShopStore(): Store
    {
        $store = null;
        if ($this->appState->getAreaCode() == Area::AREA_ADMINHTML) {
            list($scope, $scopeId) = $this->getAdminReqScopeAndId();

            if ($scope == ScopeInterface::SCOPE_WEBSITE) {
                $store = $this->storeManager->getWebsite($scopeId)->getDefaultStore();
            }
            if ($scope == ScopeInterface::SCOPE_STORE) {
                $store = $this->storeManager->getStore($scopeId);
            }
            if ($scope == ScopeConfigInterface::SCOPE_TYPE_DEFAULT) {
                $store = $this->storeManager->getDefaultStoreView();
            }
        } else {
            $store = $this->storeManager->getStore();
        }

        return $store;
    }

    public function getShopDomain($inclStoreView = true): string
    {
        $pbfInstallScope = $this->scopeConfig->getValue(StoreConfigProvider::XML_PATH_PAYBYFINANCE_INSTALL_SCOPE);
        //When on admin, it may not been saved yet:
        if ($pbfInstallScope == null && $this->appState->getAreaCode() == Area::AREA_ADMINHTML) {
            list($scopeType, $scopeId) = $this->getAdminReqScopeAndId();
            $pbfInstallScope = $scopeType;
        }

        //When PBF is installed on "for all" level, do not add storecode to shop domain:
        if ($pbfInstallScope == ScopeConfigInterface::SCOPE_TYPE_DEFAULT) {
            $inclStoreView = false;
        }

        $store = $this->getShopStore();
        if ($store == null) {
            throw new \Exception("Cannot find store view for scope.");
        }

        return $this->getStoreReference($store, $inclStoreView);
    }

    public function getStoreReference($store, $inclStoreView = true)
    {
        $path = 'web/unsecure/base_url';
        $url = $this->scopeConfig->getValue($path, ScopeInterface::SCOPE_STORE, $store->getId());
        $ref = parse_url($url, PHP_URL_HOST);
        if ($inclStoreView) {
            $ref = $ref . "/" . $store->getCode();
        }
        return $ref;
    }

    public function getPbfServerVersion(): string
    {
        return $this->getWidgetSettings()['Versions'] ?? '2.0.0.0';
    }

    public function getWidgetSettings(): ?array
    {
        if ($this->appState->getAreaCode() != Area::AREA_ADMINHTML) {
            throw new \Exception("cannot load PBF settings from FE");
        }

        if (!$this->widgetSettings) {
            list($scopeType, $scopeId) = $this->getAdminReqScopeAndId();
            $settings = $this->storeConfigProvider->getSettings($scopeType, $scopeId);
            $settings = $settings['value'] ?? '';

            try {
                $this->widgetSettings = $this->serializer->unserialize($settings);
            } catch (\Exception $e) {
                $this->widgetSettings = [];
            }
        }

        return $this->widgetSettings;
    }

    public function getAdminReqScopeAndId($resolveToDefaultStoreview = false): array
    {
        $websiteId = $this->request->getParam(ScopeInterface::SCOPE_WEBSITE);
        if ($websiteId) {
            if ($resolveToDefaultStoreview) {
                return [ScopeConfigInterface::SCOPE_TYPE_DEFAULT, 0];
            }
            return [ScopeInterface::SCOPE_WEBSITE, (int)$websiteId];
        }

        $storeId = $this->request->getParam(ScopeInterface::SCOPE_STORE);
        if ($storeId) {
            return [ScopeInterface::SCOPE_STORE, (int)$storeId];
        }

        //Why are the constants on different classes ?
        return [ScopeConfigInterface::SCOPE_TYPE_DEFAULT, 0];

    }

    /**
     * Admin only
     *
     * @return void
     */
    public function getInstallationPoint()
    {
        list($scope, $scopeId) = $this->getAdminReqScopeAndId();
        $pbfScope = $this->scopeConfig->getValue(StoreConfigProvider::XML_PATH_PAYBYFINANCE_INSTALL_SCOPE);
        if ($pbfScope == null) {
            $pbfScope = $scope;
        }
        $treeLevel = [
            ScopeConfigInterface::SCOPE_TYPE_DEFAULT => 0,
            ScopeInterface::SCOPE_WEBSITE => 1,
            ScopeInterface::SCOPE_STORE => 2
        ];

        if ($treeLevel[$scope] != $treeLevel[$pbfScope]) {
            throw new \Exception("Cannot install to " . $scope . " level. Pbf was installed on " . $pbfScope . " level.");
        }

        return [$pbfScope, $scopeId];
    }

    public function isPBFIntegrationCreated(): bool
    {
        $integrationName = StoreConfigProvider::PBF_INTEGRATION_IDENTIFIER;
        $integration = $this->integrationService->findByName($integrationName);
        return $integration && (bool)$integration->getStatus();
    }

    public function isPayByFinanceEnabled(): bool
    {
        return $this->storeConfigProvider->isPayByFinanceEnabled(\Magento\Store\Model\ScopeInterface::SCOPE_STORE);
    }

    public function getPBFIntegrationId(): int
    {
        $integrationName = StoreConfigProvider::PBF_INTEGRATION_IDENTIFIER;
        $integration = $this->integrationService->findByName($integrationName);
        return (int)$integration->getId();
    }

}
