<?php

namespace App\Master\Services;

use App\Services\AutoTraderApi;

class AutoTraderTaxonomyService
{
    /**
     * Constructor
     *
     * @param AutoTraderApi $api
     */
    public function __construct(protected AutoTraderApi $api)
    {
        //
    }

    /**
     * Get list of makes available on AutoTrader
     *
     * @return array
     */
    public function getMakes(): array
    {
        $carParams = [
            'advertiserId' => config('services.auto-trader.merchant_id'),
            'vehicleType' => 'Car',
        ];
        $vanParams = [
            'advertiserId' => config('services.auto-trader.merchant_id'),
            'vehicleType' => 'Van',
        ];
        $bikeParams = [
            'advertiserId' => config('services.auto-trader.merchant_id'),
            'vehicleType' => 'Bike',
        ];
        $caravanParams = [
            'advertiserId' => config('services.auto-trader.merchant_id'),
            'vehicleType' => 'Caravan',
        ];
        $motorhomeParams = [
            'advertiserId' => config('services.auto-trader.merchant_id'),
            'vehicleType' => 'Motorhome',
        ];

        $cars = $this->getMakesForVehicleType('cars', $carParams);
        $vans = $this->getMakesForVehicleType('lcvs', $vanParams);
        $bikes = $this->getMakesForVehicleType('motorcycles', $bikeParams);
        $caravans = $this->getMakesForVehicleType('caravans', $caravanParams);
        $motorhomes = $this->getMakesForVehicleType('motorhomes', $motorhomeParams);

        return collect(array_merge($cars, $vans, $bikes, $caravans, $motorhomes))
            ->map(fn ($make) => array_merge(
                $make,
                [
                    'cars' => array_key_exists($make['makeId'], $cars),
                    'lcvs' => array_key_exists($make['makeId'], $vans),
                    'motorcycles' => array_key_exists($make['makeId'], $bikes),
                    'caravans' => array_key_exists($make['makeId'], $caravans),
                    'motorhomes' => array_key_exists($make['makeId'], $motorhomes),
                ]
            ))->toArray();
    }

    /**
     * Get list of make models available on AutoTrader
     *
     * @return array
     */
    public function getModels(string $make_id, string $type): array
    {
        $params = [
            'advertiserId' => config('services.auto-trader.merchant_id'),
            'vehicleType' => $type,
            'makeId' => $make_id
        ];
        return collect($this->api->get("taxonomy/models?" . $this->params($params)))
            ->flatten(1)
            ->toArray();
    }

    public function getBodyStyles(string $type): array
    {
        $params = [
            'advertiserId' => config('services.auto-trader.merchant_id'),
            'vehicleType' => $type,
        ];

        return collect($this->api->get('taxonomy/bodyTypes?' . $this->params($params)))
            ->flatten(1)
            ->toArray();
    }

    protected function getMakesForVehicleType(string $vehicle_type, array $params): array
    {
        return collect($this->api->get("taxonomy/makes?" . $this->params($params)))
            ->flatten(1)
            ->mapWithKeys(fn ($vehicle) => [
                $vehicle['makeId'] => array_merge($vehicle, [$vehicle_type => true])
            ])
            ->toArray();
    }

    /**
     * Convert params to a query string
     *
     * @param array $params
     * @return string
     */
    private function params(array $params): string
    {
        return collect($params)
            ->map(function ($value, $key) {
                return $key . '=' . $value;
            })->implode('&');
    }
}
