<?php

namespace Tests\Tenant;

use App\Facades\Settings;
use App\Master\Models\VehicleMake;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleFinance;
use Mtc\VehicleSavings\SavingsCalculatorDefault;
use Mtc\VehicleSavings\SavingsCalculatorPeterVardy;
use Mtc\VehicleSavings\VehicleSavingsHelper;
use Tests\TenantTestCase;

class SavingsTest extends TenantTestCase
{

    public function testSetting()
    {
        $this->enableSavings();
        $this->assertTrue(Settings::get('vehicle-savings-enabled'));
    }

    public function testSavingsNotEnabled()
    {
        $vehicle = Vehicle::factory()->create([
            'price' => 10000,
            'original_price' => 12000,
            'rrp_price' => 18000,
            'is_vat_applicable' => true
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle);

        // test no setting
        $this->assertCount(0, $savings);

        $this->enableSavings(false);

        $this->assertEquals(false, Settings::get('vehicle-savings-enabled'));

        $vehicle = Vehicle::factory()->create([
            'price' => 10000,
            'original_price' => 12000,
            'rrp_price' => 18000,
            'is_vat_applicable' => true
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle);

        // test a disabled setting
        $this->assertCount(0, $savings);
    }

    public function testSavingsEnabled()
    {
        $this->enableSavings();
        $this->enablePreviousPrice();
        $this->enableRRPSavings();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        $vehicle = Vehicle::factory()->create([
            'price' => 100,
            'original_price' => 120,
            'rrp_price' => 180,
            'is_vat_applicable' => true
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertGreaterThan(0, $savings);
        $this->assertEquals(20, $savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertEquals(20, $savings['saving_total']);
    }

    public function testRRP()
    {
        $this->enableSavings();
        $this->enablePreviousPrice();
        $this->enableRRPSavings();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        $vehicle = Vehicle::factory()->create([
            'price' => 100,
            'original_price' => 0,
            'rrp_price' => 180,
            'is_vat_applicable' => true
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertGreaterThan(0, $savings);
        $this->assertNull($savings['saving_original_price']);
        $this->assertEquals(80, $savings['saving_rrp']);
        $this->assertEquals(80, $savings['saving_total']);
    }

    public function testLimits()
    {
        $this->enableSavings();

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'max saving',
            'config_key' => 'vehicle-savings-ceiling',
            'value' => '500',
            'type' => 'int',
        ]);

        $this->enablePreviousPrice();
        $this->enableRRPSavings();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        $vehicle = Vehicle::factory()->create([
            'price' => 100,
            'original_price' => 80,
            'rrp_price' => 789,
            'is_vat_applicable' => true
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertEquals(0, $savings['saving_original_price']);
        $this->assertEquals(500, $savings['saving_rrp']);
    }

    public function testFeatureFlagCalculationTypes()
    {
        $this->enablePVSavingsMode();

        $this->assertInstanceOf(SavingsCalculatorPeterVardy::class, VehicleSavingsHelper::getEnabledCalculator());
    }

    public function testPeterVardySavingsBestBuy()
    {
        $this->enableSavings();
        $this->enablePreviousPrice();
        $this->enableRRPSavings();
        $this->setFinanceSavingSetting(25);

        $this->enablePVSavingsMode();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'price' => 22000,
            'original_price' => 25000,
            'previous_price' => 25000,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-6 months'))
        ]);

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $expected_finance_rate_saving = 25 * ($finance_quote->total_credit_amount / 1000);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNotNull($savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertNotNull($savings['saving_finance_rate']);
        $this->assertNull($savings['saving_finance_contribution']);

        $this->assertEquals(3000, $savings['saving_original_price']);
        $this->assertEquals($expected_finance_rate_saving, $savings['saving_finance_rate']);
    }

    public function testPeterVardySavingsPremium()
    {
        $this->enableSavings();
        $this->setFinanceSavingSetting(25);

        $this->enablePreviousPrice();
        $this->enableRRPSavings();
        $this->enablePVSavingsMode();
        $this->setFinanceContributionSettings();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 22000,
            'original_price' => 25000,
            'previous_price' => 25000,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $vehicle->make_id = $make->id;
        $vehicle->type = 'car';
        $vehicle->save();
        $vehicle->refresh();

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $expected_finance_rate_saving = 25 * ($finance_quote->total_credit_amount / 1000);
        $expected_finance_contribution = 0.02 * $finance_quote->total_credit_amount;

        $this->assertEquals('peter-vardy', Settings::get('vehicle-savings-type'));

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNotNull($savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertNotNull($savings['saving_finance_rate']);
        $this->assertNotNull($savings['saving_finance_contribution']);

        $this->assertEquals(3000, $savings['saving_original_price']);
        $this->assertEquals($expected_finance_rate_saving, $savings['saving_finance_rate']);
        $this->assertEquals($expected_finance_contribution, $savings['saving_finance_contribution']);
    }

    public function testPeterVardySavingsPremiumElectric()
    {
        $this->enableSavings();
        $this->setFinanceSavingSetting(25);
        $this->setFinanceContributionSettings();
        $this->enablePreviousPrice();
        $this->enableRRPSavings();
        $this->enablePVSavingsMode();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $fuel_type = FuelType::factory()->create([
            'name' => 'electric'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 22000,
            'original_price' => 25000,
            'previous_price' => 25000,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $vehicle->make_id = $make->id;
        $vehicle->fuel_type_id = $fuel_type->id;
        $vehicle->type = 'car';
        $vehicle->save();
        $vehicle->refresh();

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $expected_finance_rate_saving = 25 * ($finance_quote->total_credit_amount / 1000);

        // electric uses a different percentage.
        // amount_to_finance * 0.05
        // capped at 1000
        $expected_finance_contribution = 1000;

        $this->enablePVSavingsMode();

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNotNull($savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertNotNull($savings['saving_finance_rate']);
        $this->assertNotNull($savings['saving_finance_contribution']);

        $this->assertEquals(3000, $savings['saving_original_price']);
        $this->assertEquals($expected_finance_rate_saving, $savings['saving_finance_rate']);
        $this->assertEquals($expected_finance_contribution, $savings['saving_finance_contribution']);
    }

    public function testPeterVardySavingsValueRange()
    {
        $this->enableSavings();
        $this->enablePreviousPrice();
        $this->enableRRPSavings();
        $this->enablePVSavingsMode();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'VW'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 9000,
            'original_price' => 9800,
            'previous_price' => 9800,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $vehicle->make_id = $make->id;
        $vehicle->type = 'car';
        $vehicle->save();
        $vehicle->refresh();

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNotNull($savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertNull($savings['saving_finance_rate']);
        $this->assertNull($savings['saving_finance_contribution']);

        $this->assertEquals(800, $savings['saving_original_price']);
    }

    public function testPeterVardySavingsCoreRange()
    {
        $this->enableSavings();
        $this->setFinanceSavingSetting(25);
        $this->setFinanceContributionSettings();
        $this->enablePreviousPrice();
        $this->enableRRPSavings();
        $this->enablePVSavingsMode();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'VW'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 18000,
            'original_price' => 0,
            'rrp_price' => 20000,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $vehicle->make_id = $make->id;
        $vehicle->type = 'car';
        $vehicle->save();
        $vehicle->refresh();

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $expected_finance_rate_saving = 25 * ($finance_quote->total_credit_amount / 1000);
        $expected_finance_contribution = 0.02 * $finance_quote->total_credit_amount;


        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNull($savings['saving_original_price']);
        $this->assertNotNull($savings['saving_rrp']);
        $this->assertNotNull($savings['saving_finance_rate']);
        $this->assertNotNull($savings['saving_finance_contribution']);

        $this->assertEquals(2000, $savings['saving_rrp']);
        $this->assertEquals($expected_finance_rate_saving, $savings['saving_finance_rate']);
        $this->assertEquals($expected_finance_contribution, $savings['saving_finance_contribution']);
    }

    public function testPeterVardySavingsVATExcluded()
    {
        $this->enableSavings();
        $this->enablePreviousPrice();
        $this->enableRRPSavings();
        $this->enablePVSavingsMode();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'VW'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 9000,
            'original_price' => 9800,
            'previous_price' => 9800,
            'rrp_price' => 0,
            'is_vat_applicable' => false,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $vehicle->make_id = $make->id;
        $vehicle->type = 'car';
        $vehicle->save();
        $vehicle->refresh();

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNotNull($savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertNull($savings['saving_finance_rate']);
        $this->assertNull($savings['saving_finance_contribution']);

        $expected_cash_saving = $vehicle->original_price - ($vehicle->price / 1.2);
        $this->assertEquals($expected_cash_saving, $savings['saving_original_price']);
    }

    public function testPeterVardySavingsPremiumSavingCap()
    {
        $this->enableSavings();
        $this->setFinanceSavingSetting(25);
        $this->setFinanceContributionSettings();
        $this->enablePVSavingsMode();

        $this->assertEquals(true, Settings::get('vehicle-savings-enabled'));

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 99000,
            'original_price' => 0,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $vehicle->make_id = $make->id;
        $vehicle->type = 'car';
        $vehicle->save();
        $vehicle->refresh();

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 1000,
            'total_credit_amount' => 111000
        ]);

        $expected_finance_rate_saving = 25 * ($finance_quote->total_credit_amount / 1000);

        // capped at £600 for combustion engines
        $expected_finance_contribution = 600;

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNull($savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertNotNull($savings['saving_finance_rate']);
        $this->assertNotNull($savings['saving_finance_contribution']);

        $this->assertEquals($expected_finance_rate_saving, $savings['saving_finance_rate']);
        $this->assertEquals($expected_finance_contribution, $savings['saving_finance_contribution']);
    }

    public function testCheapestFinanceExample()
    {
        $vehicle = Vehicle::factory()->create();

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 111
        ]);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 5
        ]);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 222
        ]);

        $default = $vehicle->defaultFinanceExample;

        $this->assertNotNull($default);
        $this->assertEquals(111, $vehicle->financeExamples->first()->monthly_price);
        $this->assertEquals(5, $default->monthly_price);
    }

    public function testSavingsDisabled()
    {
        $this->enableSavings();
        $this->enablePVSavingsMode();

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 19000,
            'original_price' => 21000,
            'rrp_price' => 29000,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $vehicle->make_id = $make->id;
        $vehicle->type = 'car';
        $vehicle->save();
        $vehicle->refresh();

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertArrayHasKey('saving_original_price', $savings);
        $this->assertArrayHasKey('saving_rrp', $savings);
        $this->assertArrayHasKey('saving_finance_rate', $savings);
        $this->assertArrayHasKey('saving_finance_contribution', $savings);

        $this->assertNull($savings['saving_original_price']);
        $this->assertNull($savings['saving_rrp']);
        $this->assertNull($savings['saving_finance_rate']);
        $this->assertNull($savings['saving_finance_contribution']);
    }

    public function testFinanceSavingDisabled()
    {
        $this->enableSavings();
        $this->enablePVSavingsMode();

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 22000,
            'original_price' => 25000,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertNull($savings['saving_finance_rate']);
    }

    public function testFinanceSavingEnabled()
    {
        $this->enableSavings();
        $this->enablePVSavingsMode();
        $this->setFinanceSavingSetting(25);

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 22000,
            'original_price' => 25000,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertNotNull($savings['saving_finance_rate']);
    }

    public function testFinanceContributionDisabled()
    {
        $this->enableSavings();
        $this->enablePVSavingsMode();

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 22000,
            'original_price' => 25000,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertNull($savings['saving_finance_contribution']);
    }

    public function testFinanceContributionEnabled()
    {
        $this->enableSavings();
        $this->enablePVSavingsMode();
        $this->setFinanceContributionSettings();

        Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi'
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => 1,
            'type' => 'car',
            'price' => 22000,
            'original_price' => 25000,
            'rrp_price' => 0,
            'is_vat_applicable' => true,
            'first_registration_date' => date('Y-m-d', strtotime('-24 months'))
        ]);

        $finance_quote = VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'monthly_price' => 200,
            'total_credit_amount' => 24000
        ]);

        $savings = VehicleSavingsHelper::getEnabledCalculator()->getSavings($vehicle)->toArray();

        $this->assertNotNull($savings['saving_finance_contribution']);
    }

    public function testDefaultSavingsMode()
    {
        $this->assertInstanceOf(SavingsCalculatorDefault::class, VehicleSavingsHelper::getEnabledCalculator());
    }

    private function enableSavings($enabled = true)
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled',
            'value' => $enabled,
            'type' => 'bool',
        ]);
    }

    private function enablePreviousPrice()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled-previous-price',
            'value' => true,
            'type' => 'bool',
        ]);
    }

    private function enableRRPSavings()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled-rrp',
            'value' => true,
            'type' => 'bool',
        ]);
    }

    private function enablePVSavingsMode()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-type',
            'value' => 'peter-vardy',
            'type' => 'string',
        ]);
    }

    private function setFinanceSavingSetting($value = 1)
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled-finance-saving',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-pv-finance-saving-pounds-per-thousand',
            'value' => $value,
            'type' => 'int',
        ]);
    }

    private function setFinanceContributionSettings()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled-finance-contribution',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-pv-finance-contribution-percent-ice',
            'value' => 2,
            'type' => 'int',
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-pv-finance-contribution-percent-electric',
            'value' => 5,
            'type' => 'int',
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-pv-finance-contribution-max-ice',
            'value' => 600,
            'type' => 'int',
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-pv-finance-contribution-max-electric',
            'value' => 1000,
            'type' => 'int',
        ]);
    }
}
