<?php

namespace App\Exports;

use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class CarGurusExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::CAR_GURUS),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::CAR_GURUS)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'vrm' => $vehicle->registration_number,
                    'make' => $vehicle->make?->name,
                    'model' => $vehicle->model?->name,
                    'year' => $vehicle->manufacture_year,
                    'variant' => $vehicle->derivative,
                    'price' => $vehicle->price,
                    'mileage' => $vehicle->odometer_mi,
                    'image_urls' => $vehicle->mediaUses
                        ->filter(fn($mediaUse) => $mediaUse->media)
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large'))
                        ->implode(','),
                    'colour' => $vehicle->colour,
                    'comments' => '',
                    'stock_number' => $vehicle->uuid,
                    'gearbox' => $vehicle->transmission?->name,
                    'installed_options' => $vehicle->specs
                        ->pluck('description')
                        ->implode(','),
                    'dealer_id' => $vehicle->dealership?->data['car-guru-dealer-id'] ?? null,
                    'dealer_name' => $vehicle->dealership?->name,
                    'dealer_street_address' => $vehicle->dealership?->address1,
                    'dealer_county' => $vehicle->dealership?->county,
                    'dealer_postcode' => $vehicle->dealership?->postcode,
                    'dealer_crm_email' => $vehicle->dealership?->email,
                    'registration_date' => $vehicle->first_registration_date,
                    'is_new' => (int)$vehicle->is_new,
                    'doors' => $vehicle->door_count,
                    'engine_size' => $vehicle->engine_size_cc,
                    'fuel_type' => $vehicle->fuelType?->name,
                    'plus_vat' => $vehicle->type === VehicleType::LCV ? 0 : 1
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'vrm',
            'make',
            'model',
            'year',
            'variant',
            'price',
            'mileage',
            'image_urls',
            'colour',
            'comments',
            'stock_number',
            'gearbox',
            'installed_options',
            'dealer_id',
            'dealer_name',
            'dealer_street_address',
            'dealer_county',
            'dealer_postcode',
            'dealer_crm_email',
            'registration_date',
            'is_new',
            'doors',
            'engine_size',
            'fuel_type',
            'plus_vat'
        ];
    }
}
