<?php

namespace App\Imports;

use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Mtc\MercuryDataModels\SeoRedirect;

class SeoRedirectImport implements ToCollection, WithHeadingRow
{
    public function collection(Collection $collection): void
    {
        // Normalize all rows first (strip trailing slashes)
        $normalizedRows = $collection->map(fn ($row) => [
            'from' => rtrim($row['from'] ?? '', '/'),
            'to' => rtrim($row['to'] ?? '', '/'),
            'from_domain' => $row['from_domain'] ?? null,
            'to_domain' => $row['to_domain'] ?? null,
            'code' => $row['code'] ?? 301,
            'has_regex' => $row['has_regex'] ?? 0,
        ]);

        // Build redirect map from import data (from => to)
        $importRedirectMap = $normalizedRows
            ->keyBy('from')
            ->map(fn ($row) => $row['to']);

        // Get existing redirects from database
        $existingRedirects = SeoRedirect::query()
            ->pluck('to', 'from')
            ->mapWithKeys(fn ($to, $from) => [rtrim($from, '/') => rtrim($to, '/')]);

        // Merge maps: import takes precedence (will overwrite existing)
        $fullRedirectMap = $existingRedirects->merge($importRedirectMap);

        // Import valid redirects
        foreach ($normalizedRows as $row) {
            if ($this->wouldCreateLoop($row['from'], $row['to'], $row, $fullRedirectMap)) {
                continue;
            }

            SeoRedirect::query()
                ->updateOrCreate([
                    'from' => $row['from'],
                    'from_domain' => $row['from_domain'],
                ], [
                    'to' => $row['to'],
                    'code' => $row['code'],
                    'to_domain' => $row['to_domain'],
                    'has_regex' => $row['has_regex'],
                ]);
        }
    }

    /**
     * Check if adding this redirect would create a loop
     * Follows the redirect chain to detect cycles
     */
    private function wouldCreateLoop(string $from, string $to, array $row, Collection $redirectMap): bool
    {
        // Self-redirect check
        if ($from === $to && ($row['from_domain'] === $row['to_domain'] || empty($row['to_domain']))) {
            return true;
        }

        // Follow the chain from 'to' to see if we ever get back to 'from'
        $visited = [$from => true];
        $current = $to;
        $maxDepth = 50; // Prevent infinite loops in case of bad data

        for ($i = 0; $i < $maxDepth; $i++) {
            // If we've seen this path before, it's a loop
            if (isset($visited[$current])) {
                return true;
            }

            // If there's no further redirect, chain ends - no loop
            if (!$redirectMap->has($current)) {
                return false;
            }

            $visited[$current] = true;
            $current = $redirectMap->get($current);
        }

        // Max depth reached - treat as potential loop
        return true;
    }
}
