<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;
use Symfony\Component\HttpFoundation\Response;

class DealershipAccessMiddleware
{
    /**
     * Map of route parameter names to their corresponding model classes.
     */
    protected array $modelMap = [
        'vehicle' => Vehicle::class,
        'vehicle_offer' => VehicleOffer::class,
        'vehicleOffer' => VehicleOffer::class,
        'new_car' => NewCar::class,
        'newCar' => NewCar::class,
        'dealership' => Dealership::class,
    ];

    /**
     * Handle an incoming request.
     *
     * @param string|null $modelParam Route parameter name (e.g., 'vehicle', 'vehicle_offer')
     */
    public function handle(Request $request, Closure $next, ?string $modelParam = null): Response
    {
        $user = $request->user();

        if (!$user) {
            abort(403, 'Unauthorized');
        }

        if (!$modelParam) {
            return $next($request);
        }

        $model = $this->resolveModel($request, $modelParam);

        if (!$model) {
            return $next($request);
        }

        $dealershipId = $this->getDealershipId($model);

        if ($dealershipId && !$user->hasAccessToDealership($dealershipId)) {
            abort(403, 'You do not have access to this dealership');
        }

        return $next($request);
    }

    /**
     * Resolve the model from route parameter.
     * Handles both route model binding (Model instance) and raw IDs.
     */
    protected function resolveModel(Request $request, string $modelParam): ?Model
    {
        $value = $request->route($modelParam);

        if (!$value) {
            return null;
        }

        // Already resolved via route model binding
        if ($value instanceof Model) {
            return $value;
        }

        // Raw ID - resolve from database
        $modelClass = $this->modelMap[$modelParam] ?? null;

        if (!$modelClass) {
            return null;
        }

        return $modelClass::query()->withTrashed()->find($value);
    }

    /**
     * Get the dealership ID from the model.
     */
    protected function getDealershipId(Model $model): ?int
    {
        // If it's a Dealership model, use its own ID
        if ($model instanceof Dealership) {
            return $model->id;
        }

        // Otherwise, check for dealership_id attribute
        if (isset($model->dealership_id)) {
            return $model->dealership_id;
        }

        return null;
    }
}
