<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Mtc\MercuryDataModels\Setting;

class PrepareTestingDatabases extends Command
{
    protected $signature = 'testing:prepare-databases {--master-only} {--tenant-only} {--with-tenant-seeders}';
    protected $description = 'Build template master and tenant SQLite databases.';

    public function handle()
    {
        config(['app.key' => 'base64:/P9JbApgF7Ld06dZDwX+JdnThiTitCBHtatZvfZCpBg=']);
        $base = base_path('database/testing');

        if (!is_dir($base) && !mkdir($base)) {
            throw new \RuntimeException(sprintf('Directory "%s" was not created', $base));
        }

        // Remove ALL previous test db files in this folder
        foreach (glob($base . '/*') as $file) {
            if (is_file($file)) {
                @unlink($file);
            }
        }

        $masterOnly = $this->option('master-only');
        $tenantOnly = $this->option('tenant-only');

        if (!$tenantOnly) {
            $this->prepareMasterDatabase($base);
        }

        if (!$masterOnly) {
            $this->prepareTenantDatabase($base);
        }

        return Command::SUCCESS;
    }

    private function prepareMasterDatabase(string $base): void
    {
        $masterFile = $base . '/master-template.sqlite';
        $schemaPath = base_path('database/schema/test-master-schema.sql');

        if (!file_exists($schemaPath)) {
            $this->error("Master schema not found at: $schemaPath");
            return;
        }

        config()->set('database.connections.master-test', [
            'driver' => 'sqlite',
            'database' => ':memory:',
            'foreign_key_constraints' => true,
        ]);

        Config::set('database.default', 'master-test');
        Config::set('tenancy.database.central_connection', 'master-test');

        // Use migrate with schema-path (same approach as tenant)
        // Schema contains all tables + migrations records, then new migrations run on top
        Artisan::call('migrate', [
            '--database' => 'master-test',
            '--schema-path' => $schemaPath,
            '--path' => database_path('migrations'),
            '--realpath' => true,
            '--force' => true,
        ]);

        Setting::unsetEventDispatcher();
        // Run global seeders individually (skip CurrencyService which makes HTTP calls)
        $globalSeeders = [
//            \Database\Seeders\Global\GlobalRoleSeed::class,
//            \Database\Seeders\Global\PrimaryBillableSeed::class,
//            \Database\Seeders\Global\VehicleTaxonomySeeder::class,
//            \Database\Seeders\Global\CountrySeeder::class,
        ];

        foreach ($globalSeeders as $seederClass) {
            try {
                (new $seederClass())->run();
            } catch (\Throwable $e) {
                $this->error("{$seederClass} error: " . $e->getMessage());
            }
        }

        DB::connection('master-test')->getPdo()->exec("VACUUM INTO '$masterFile'");

        $this->info("Master template created: $masterFile");
    }

    private function prepareTenantDatabase(string $base): void
    {
        $tenantFile = $base . '/tenant-template.sqlite';

        config()->set('database.connections.tenant-test', [
            'driver' => 'sqlite',
            'database' => ':memory:',
            'foreign_key_constraints' => true,
        ]);

        Config::set('database.default', 'tenant-test');

        Artisan::call('migrate', [
            '--database' => 'tenant-test',
            '--schema-path' => database_path('schema/test-tenant-schema.sql'),
            '--path' => database_path('migrations/tenant'),
            '--realpath' => true,
            '--force' => true,
        ]);

        // Use seeders if --with-tenant-seeders flag is passed
        if ($this->option('with-tenant-seeders')) {
            Setting::unsetEventDispatcher();
            $tenantSeeders = glob(base_path('database/seeders/Tenant/*.php'));
            foreach ($tenantSeeders as $file) {
                $seederClass = 'Database\\Seeders\\Tenant\\' . pathinfo($file, PATHINFO_FILENAME);

                try {
                    (new $seederClass())->run();
                } catch (\Throwable $e) {
                    $this->error($e->getMessage());
                }
            }
        }

        DB::connection('tenant-test')->getPdo()->exec("VACUUM INTO '$tenantFile'");

        $this->info("Tenant template created: $tenantFile");
    }
}
