<?php

namespace Tests\Feature\Controllers;

use App\Models\ReleaseNote;
use Tests\DatabaseTestCase;
use Tests\UserForTenant;

class ReleaseNoteControllerTest extends DatabaseTestCase
{
    use UserForTenant;

     /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        ReleaseNote::factory(3)->create();

        $response = $this->actingAs($this->getUser())
            ->getJson(route('release-notes.index', [], false));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertCount(3, $response->json('data'));
    }

    public function testStore()
    {
        $response = $this->actingAs($this->getUser())
            ->postJson(route('release-notes.store', [], false), [
                'title' => 'foo'
            ]);

        $response->assertStatus(201);
        $this->assertEquals('foo', $response->json('title'));
        $this->assertTrue(ReleaseNote::query()->where('title', 'foo')->exists());

    }

    public function testUpdate()
    {
        $releaseNote = ReleaseNote::factory()->create(['title' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->putJson(route('release-notes.update', $releaseNote->id, false), [
                'is_published' => true,
                'published_at' => '2025-05-01',
                'title' => 'baz',
                'content' => 'lorem ipsum dolor',
            ]);

        $releaseNote->refresh();
        $response->assertStatus(200);
        $this->assertEquals('baz', $releaseNote->title);
        $this->assertEquals('lorem ipsum dolor', $releaseNote->content);
        $this->assertEquals('2025-05-01', $releaseNote->published_at->format('Y-m-d'));
        $this->assertTrue($releaseNote->is_published);

    }

    public function testShow()
    {
        $releaseNote = ReleaseNote::factory()->create(['title' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->getJson(route('release-notes.show', $releaseNote->id, false));

        $response->assertStatus(200);
        $this->assertEquals($releaseNote->id, $response->json('id'));
        $this->assertEquals('foo', $response->json('title'));

    }

    public function testDelete()
    {
        $releaseNote = ReleaseNote::factory()->create(['title' => 'foo']);
        $response = $this->actingAs($this->getUser())
            ->deleteJson(route('release-notes.destroy', $releaseNote->id, false));

        $response->assertStatus(200);
        $this->assertFalse(ReleaseNote::query()->where('id', $releaseNote->id)->exists());
    }
}
