<?php

namespace Tests\Feature;

use App\Jobs\SyncTaxonomyToTenants;
use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;
use Tests\DatabaseTestCase;
use Tests\UserForTenant;


class LeaseAdminTest extends DatabaseTestCase
{
    use UserForTenant;

    public function testUpdateMakeTaxonomyKeyloopId()
    {
        $vehicle_make = VehicleMake::factory()->create([
            'keyloop_id' => null,
        ]);

        $url = '/api/taxonomies/{type}/{id}';
        $url = str_replace('{type}', 'make', $url);
        $url = str_replace('{id}', $vehicle_make->id, $url);

        $response = $this->actingAs($this->getUser())
            ->putJson($url, [
                'keyloop_id' => 'foo',
            ]);

        $vehicle_make->refresh();
        $response->assertStatus(200);
        $this->assertEquals('foo', $vehicle_make->keyloop_id);
    }

    public function testUpdateModelTaxonomyKeyloopId()
    {
        $vehicle_model = VehicleModel::factory()->create([
            'keyloop_id' => null,
        ]);

        $url = '/api/taxonomies/{type}/{id}';
        $url = str_replace('{type}', 'model', $url);
        $url = str_replace('{id}', $vehicle_model->id, $url);

        $response = $this->actingAs($this->getUser())
            ->putJson($url, [
                'keyloop_id' => 'foo',
            ]);

        $vehicle_model->refresh();
        $response->assertStatus(200);
        $this->assertEquals('foo', $vehicle_model->keyloop_id);
    }

    public function testKeyloopIdTaxonomySync()
    {
        $master_make = VehicleMake::factory()->create([
            'keyloop_id' => 'foo',
        ]);

        $master_model = VehicleModel::factory()->create([
            'keyloop_id' => 'bar',
        ]);

        $tenant_make = \Mtc\MercuryDataModels\VehicleMake::factory()->create([
            'id' => $master_make->id,
            'keyloop_id' => null,
        ]);

        $tenant_model = \Mtc\MercuryDataModels\VehicleModel::factory()->create([
            'id' => $master_model->id,
            'keyloop_id' => null,
        ]);

        (new SyncTaxonomyToTenants('make', $master_make))->handle();
        (new SyncTaxonomyToTenants('model', $master_model))->handle();

        $tenant_make->refresh();
        $tenant_model->refresh();

        $this->assertEquals('foo', $tenant_make->keyloop_id);
        $this->assertEquals('bar', $tenant_model->keyloop_id);
    }

    public function testListMakes()
    {
        \Mtc\MercuryDataModels\VehicleMake::factory(5)->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.leasing.makes.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, $response->json('total'));
    }

    public function testShowMake()
    {
        $make = \Mtc\MercuryDataModels\VehicleMake::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.leasing.makes.show', $make->id));

        $response->assertStatus(200);

        $this->assertEquals($make->name, $response->json('name'));
    }

    public function testListModels()
    {
        $make = \Mtc\MercuryDataModels\VehicleMake::factory()->create();
        \Mtc\MercuryDataModels\VehicleModel::factory(3)->create([
            'make_id' => $make->id,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.leasing.models.index', $make->id));

        $response->assertStatus(200);
        $this->assertEquals(3, $response->json('total'));
    }

    public function testInvalidMakeUpdateRequest()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.makes.update-bulk'));

        $response->assertStatus(422);
        $this->assertArrayHasKey('errors', $response->json());
        $this->assertArrayHasKey('makes', $response->json('errors'));

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.makes.update-bulk'), [
                'makes' => [
                    [],
                    [],
                ],
            ]);

        $response->assertStatus(422);
        $this->assertArrayHasKey('errors', $response->json());
        $this->assertArrayHasKey('makes.0.id', $response->json('errors'));
        $this->assertArrayHasKey('makes.0.keyloop_import_enabled', $response->json('errors'));
    }

    public function testMakeBulkUpdate()
    {
        $make_1 = \Mtc\MercuryDataModels\VehicleMake::factory()->create([
            'keyloop_import_enabled' => false,
        ]);

        $make_2 = \Mtc\MercuryDataModels\VehicleMake::factory()->create([
            'keyloop_import_enabled' => false,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.makes.update-bulk'), [
                'makes' => [
                    [
                        'id' => $make_1->id,
                        'keyloop_import_enabled' => true,
                    ],
                    [
                        'id' => $make_2->id,
                        'keyloop_import_enabled' => false,
                    ],
                ]
            ]);

        $response->assertStatus(200);

        $make_1->refresh();
        $make_2->refresh();

        $this->assertTrue($make_1->keyloop_import_enabled);
        $this->assertFalse($make_2->keyloop_import_enabled);
    }

    public function testInvalidModelUpdateRequest()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.models.update-bulk', 123));

        $response->assertStatus(422);
        $this->assertArrayHasKey('errors', $response->json());
        $this->assertArrayHasKey('models', $response->json('errors'));

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.models.update-bulk', 123), [
                'models' => [
                    [],
                    [],
                ],
            ]);

        $response->assertStatus(422);
        $this->assertArrayHasKey('errors', $response->json());
        $this->assertArrayHasKey('models.0.id', $response->json('errors'));
        $this->assertArrayHasKey('models.0.keyloop_import_enabled', $response->json('errors'));
    }

    public function testModelBulkUpdate()
    {
        $make = \Mtc\MercuryDataModels\VehicleMake::factory()->create();
        $model_1 = \Mtc\MercuryDataModels\VehicleModel::factory()->create([
            'make_id' => $make->id,
            'keyloop_import_enabled' => false,
        ]);

        $model_2 = \Mtc\MercuryDataModels\VehicleModel::factory()->create([
            'make_id' => $make->id,
            'keyloop_import_enabled' => false,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.models.update-bulk', $make->id), [
                'models' => [
                    [
                        'id' => $model_1->id,
                        'keyloop_import_enabled' => true,
                    ],
                    [
                        'id' => $model_2->id,
                        'keyloop_import_enabled' => false,
                    ],
                ]
            ]);

        $response->assertStatus(200);

        $model_1->refresh();
        $model_2->refresh();

        $this->assertTrue($model_1->keyloop_import_enabled);
        $this->assertFalse($model_2->keyloop_import_enabled);
    }

    public function testMakeUpdate()
    {
        $make = \Mtc\MercuryDataModels\VehicleMake::factory()->create([
            'keyloop_import_enabled' => true,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.leasing.makes.update', $make->id), [
                'keyloop_import_enabled' => false,
            ]);

        $response->assertStatus(200);

        $make->refresh();

        $this->assertFalse($make->keyloop_import_enabled);
    }

    public function testMakeUpdateInvalidRequest()
    {
        $make = \Mtc\MercuryDataModels\VehicleMake::factory()->create([
            'keyloop_import_enabled' => true,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.leasing.makes.update', $make->id), [
                'foo' => 'bar',
            ]);

        $response->assertStatus(422);
    }
}
