<?php

namespace Tests\Feature;

use App\Jobs\RemoveOrphanedTaxonomiesFromModelsJob;
use App\Mail\OrphanedTaxonomyRemovedMail;
use App\TaxonomyMap;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Mail;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Tenant;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\DatabaseTestCase;

class RemoveOrphanedTaxonomyMapsTest extends DatabaseTestCase
{

    public function testJobDispatch()
    {
        Bus::fake();
        Artisan::call('taxonomy:cleanup');
        Bus::assertDispatched(RemoveOrphanedTaxonomiesFromModelsJob::class);
    }

    public function testDelete()
    {
        Mail::fake();

        TaxonomyMap::query()->create([
            'taxonomy_type' => 'make',
            'taxonomy_id' => 123,
            'term' => 'foo',
            'provider' => 'bar',
        ]);

        $this->assertCount(1, TaxonomyMap::all());

        Artisan::call('taxonomy:cleanup');

        $this->assertCount(0, TaxonomyMap::all());
    }

    public function testUpdateMakeIds()
    {
        Mail::fake();

        Tenant::query()->update([
            'name' => 'test_tenant',
        ]);

        TaxonomyMap::query()->create([
            'taxonomy_type' => 'make',
            'taxonomy_id' => 123,
            'term' => 'foo',
            'provider' => 'bar',
        ]);

        TaxonomyMap::query()->create([
            'taxonomy_type' => 'model',
            'taxonomy_id' => 456,
            'term' => 'baz',
            'provider' => 'buzz',
        ]);

        Vehicle::factory()->create([
            'make_id' => 123,
            'model_id' => 456,
        ]);

        Vehicle::factory()->create([
            'make_id' => 789,
            'model_id' => 555,
        ]);

        VehicleOffer::factory()->create([
            'make_id' => 123,
            'model_id' => 456,
        ]);

        NewCar::factory()->create([
            'make_id' => 123,
            'model_id' => 456,
        ]);

        $orphaned_maps = TaxonomyMap::all();

        (new RemoveOrphanedTaxonomiesFromModelsJob($orphaned_maps))->handle();

        // assert that we set make_id and model_id to NULL for affected vehicles
        $this->assertCount(
            1,
            Vehicle::query()
                ->whereNull('make_id')
                ->whereNull('model_id')
                ->get()
        );

        // assert that we set make_id and model_id to NULL for affected vehicle offers
        $this->assertCount(
            1,
            VehicleOffer::query()
                ->whereNull('make_id')
                ->whereNull('model_id')
                ->get()
        );

        // assert that we set make_id and model_id to NULL for affected new cars
        $this->assertCount(
            1,
            NewCar::query()
                ->whereNull('make_id')
                ->whereNull('model_id')
                ->get()
        );

        // assert that we left make_id and model_id unchanged for unaffected vehicles
        $this->assertCount(
            1,
            Vehicle::query()
                ->where('make_id', 789)
                ->where('model_id', 555)
                ->get()
        );
    }

    public function testMail()
    {
        Mail::fake();

        Tenant::query()->update([
            'name' => 'test_tenant',
        ]);

        TaxonomyMap::query()->create([
            'taxonomy_type' => 'make',
            'taxonomy_id' => 123,
            'term' => 'foo',
            'provider' => 'bar',
        ]);

        TaxonomyMap::query()->create([
            'taxonomy_type' => 'model',
            'taxonomy_id' => 456,
            'term' => 'baz',
            'provider' => 'buzz',
        ]);

        Vehicle::factory()->create([
            'make_id' => 123,
            'model_id' => 456,
            'registration_number' => 'ABC123',
        ]);

        Vehicle::factory()->create([
            'make_id' => 789,
            'model_id' => 555,
        ]);

        VehicleOffer::factory()->create([
            'make_id' => 123,
            'model_id' => 456,
            'name' => 'offer_foo',
        ]);

        NewCar::factory()->create([
            'make_id' => 123,
            'model_id' => 456,
            'name' => 'new_car_bar',
        ]);

        $orphaned_maps = TaxonomyMap::all();

        (new RemoveOrphanedTaxonomiesFromModelsJob($orphaned_maps))->handle();

        Mail::assertSent(OrphanedTaxonomyRemovedMail::class, function (OrphanedTaxonomyRemovedMail $mail) {
            return $mail->assertHasSubject('Orphaned taxonomies removed report')
                && $mail->assertSeeInOrderInHtml([
                    'make taxonomy maps were removed',
                    'foo',
                    'model taxonomy maps were removed',
                    'baz',
                    'affects:',
                    'test_tenant',
                    'vehicles',
                    'ABC123',
                    'vehicle_offers',
                    'offer_foo',
                    'new_cars',
                    'new_car_bar',
                ]);
        });
    }
}
