<?php

namespace Tests\Tenant;

use App\Facades\Settings;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\DealershipDepartment;
use Tests\TenantTestCase;

class DealershipTest extends TenantTestCase
{
    public function testDepartments()
    {
        $dealership_1 = Dealership::factory()->create();
        $dealership_2 = Dealership::factory()->create();
        $department_1 = DealershipDepartment::query()->create([
            'name' => 'foo',
            'dealership_id' => $dealership_1->id,
        ]);
        $department_2 = DealershipDepartment::query()->create([
            'name' => 'baz',
            'dealership_id' => $dealership_1->id,
        ]);
        $department_3 = DealershipDepartment::query()->create([
            'name' => 'bar',
            'dealership_id' => $dealership_2->id,
        ]);

        $this->assertCount(2, Dealership::all());
        $this->assertCount(3, DealershipDepartment::all());
        $this->assertCount(2, $dealership_1->departments);
        $this->assertCount(1, $dealership_2->departments);

        $this->assertEquals($dealership_1->id, $department_1->dealership->id);
        $this->assertEquals($dealership_1->id, $department_2->dealership->id);
        $this->assertEquals($dealership_2->id, $department_3->dealership->id);
    }

    public function testDepartmentOpenTimes()
    {
        Settings::make([
            'tab' => 'automotive',
            'section' => 'image-sync',
            'group' => 'autos-on-show',
            'name' => 'enabled',
            'config_key' => 'automotive-dealership-open-time',
            'value' => '01:01',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'automotive',
            'section' => 'image-sync',
            'group' => 'autos-on-show',
            'name' => 'enabled',
            'config_key' => 'automotive-dealership-close-time',
            'value' => '23:51',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'automotive',
            'section' => 'image-sync',
            'group' => 'autos-on-show',
            'name' => 'enabled',
            'config_key' => 'automotive-dealership-open-days',
            'value' => 'all',
            'type' => 'string',
        ]);

        $dealership_with_department = Dealership::factory()->create([
            'open_times' => [],
        ]);
        $dealership_without_department = Dealership::factory()->create([
            'open_times' => $this->getOpenTimes(),
        ]);
        $dealership_for_opens_at = Dealership::factory()->create();
        $dealership_for_opens_at->update(['open_times' => []]);

        DealershipDepartment::query()->create([
            'name' => 'foo',
            'dealership_id' => $dealership_with_department->id,
            'open_times' => $this->getOpenTimes(),
            'is_primary' => true,
        ]);
        DealershipDepartment::query()->create([
            'name' => 'baz',
            'dealership_id' => $dealership_with_department->id,
            'open_times' => $this->getOpenTimes(),
        ]);
        DealershipDepartment::query()->create([
            'name' => 'bar',
            'is_primary' => true,
            'dealership_id' => $dealership_for_opens_at->id,
            'open_times' => $this->getOpenTimesForOpensAt(),
        ]);
        $dealership_for_opens_at->load('primaryDepartment');

        if (!$dealership_with_department->is_open) {
            $this->assertEquals('01:01', $dealership_with_department->opens_at['open']);
            $this->assertEquals('23:51', $dealership_with_department->opens_at['close']);
        }

        if (!$dealership_without_department->is_open) {
            $this->assertEquals('00:00', $dealership_without_department->opens_at['open']);
            $this->assertEquals('23:59', $dealership_without_department->opens_at['close']);
        }

        if (!$dealership_for_opens_at->is_open) {
            $this->assertNotNull($dealership_for_opens_at->opens_at);
            $this->assertIsArray($dealership_for_opens_at->opens_at);
            $this->assertArrayHasKey('open', $dealership_for_opens_at->opens_at);
            $this->assertArrayHasKey('close', $dealership_for_opens_at->opens_at);
            $this->assertEquals('23:58', $dealership_for_opens_at->opens_at['open']);
            $this->assertEquals('23:59', $dealership_for_opens_at->opens_at['close']);
        }
        $this->markTestSkipped();
    }

    private function getOpenTimes(): array
    {
        return [
            [
                "dayOfWeek" => 1,
                "open" => "00:00",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 2,
                "open" => "00:00",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 3,
                "open" => "00:00",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 4,
                "open" => "00:00",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 5,
                "open" => "00:00",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 6,
                "open" => "00:00",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 7,
                "open" => "00:00",
                "close" => "23:59"
            ]
        ];
    }

    private function getOpenTimesForOpensAt(): array
    {
        return [
            [
                "dayOfWeek" => 1,
                "open" => "23:58",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 2,
                "open" => "23:58",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 3,
                "open" => "23:58",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 4,
                "open" => "23:58",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 5,
                "open" => "23:58",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 6,
                "open" => "23:58",
                "close" => "23:59"
            ],
            [
                "dayOfWeek" => 7,
                "open" => "23:58",
                "close" => "23:59"
            ]
        ];
    }
}
