<?php

namespace Tests\Tenant;

use App\Facades\Site;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TenantTestCase;

final class SiteRepositoryTest extends TenantTestCase
{
    protected $tenantDomain = 'http://foo.localhost';

    public function testVehicleUrlWithAbsoluteAndAccess()
    {
        $url = Site::vehicleUrl(Vehicle::factory()->create(), true, true);

        $this->assertStringStartsWith($this->tenantDomain, $url);
        $this->assertStringContainsString('a=', $url);
    }

    public function testVehicleUrlWithoutAbsoluteAndWithAccess()
    {
        $vehicle = Vehicle::factory()->create();

        $url = Site::vehicleUrl($vehicle, true, false);

        $this->assertStringNotContainsString($this->tenantDomain, $url);
        $this->assertStringContainsString($vehicle->urlPath(), $url);
        $this->assertStringContainsString('a=', $url);
    }

    public function testVehicleUrlWithoutAbsoluteAndWithoutAccess()
    {
        $vehicle = Vehicle::factory()->create();

        $url = Site::vehicleUrl($vehicle, false, false);

        $this->assertStringNotContainsString($this->tenantDomain, $url);
        $this->assertStringContainsString($vehicle->urlPath(), $url);
    }

    public function testPageUrlForHomePage()
    {
        $url = Site::pageUrl(Page::factory()->create(['slug' => '']), false);

        $this->assertEquals($this->tenantDomain . '/', $url);
    }

    public function testPageUrlForOtherPages()
    {
        $url = Site::pageUrl(Page::factory()->create(['slug' => 'about']), false);

        $this->assertEquals($this->tenantDomain . '/about', $url);
    }

    public function testOfferUrl()
    {
        $offer = VehicleOffer::factory()->create();

        $url = Site::offerUrl($offer, true);

        $this->assertStringContainsString($offer->urlPath(), $url);
        $this->assertStringContainsString('a=', $url);
    }

    public function testNewCarUrl()
    {
        $newCar = NewCar::factory()->create();

        $url = Site::newCarUrl($newCar, false);

        $this->assertStringContainsString($newCar->urlPath(), $url);
        $this->assertStringStartsWith($this->tenantDomain, $url);
    }

    public function testGenericUrl()
    {
        $this->assertEquals($this->tenantDomain . '/foo/bar', Site::url('foo/bar'));
    }

    public function testUrlWithoutAbsolute()
    {
        $this->assertEquals('/foo/bar', Site::url('foo/bar', false));
    }

    public function testVehicleUrlWithoutWww()
    {
        $this->assertStringNotContainsString('www.', Site::vehicleUrl(Vehicle::factory()->create()));
    }
}
