<?php

namespace Tests\Traits;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Stancl\Tenancy\Bootstrappers\DatabaseTenancyBootstrapper;
use Stancl\Tenancy\Events\TenancyInitialized;

/**
 * Handles SQLite test database creation and connection configuration.
 * Single responsibility: database files and connections only.
 */
trait UsesDatabase
{
    protected bool $needsMasterDatabase = true;
    protected bool $needsTenantDatabase = true;

    private array $sqliteTestFiles = [];

    private const TYPE_MASTER = 'master';
    private const TYPE_TENANT = 'tenant';

    protected function setUpDatabase(): void
    {
        if ($this->needsMasterDatabase) {
            $this->setUpDb(self::TYPE_MASTER);
        }

        if ($this->needsTenantDatabase) {
            $this->setUpDb(self::TYPE_TENANT);
            $this->disableDatabaseTenancyBootstrapper();
            $this->registerTenancyConnectionListener();
        }

        // Master-only: point tenant connection to master
        if ($this->needsMasterDatabase && !$this->needsTenantDatabase) {
            $this->pointTenantConnectionToMaster();
        }
    }

    private function setUpDb(string $type): void
    {
        $templatePath = $this->getTemplatePath($type);
        $this->assertTemplateExists($templatePath);

        $testPath = $this->generateTestPath($type);
        copy($templatePath, $testPath);

        $this->sqliteTestFiles[$type] = $testPath;
        $this->configureConnection($type, $testPath);

        if ($type === self::TYPE_TENANT) {
            DB::purge('tenant');
        }
    }

    private function configureConnection(string $type, string $filePath): void
    {
        Config::set("database.connections.{$type}", [
            'driver' => 'sqlite',
            'database' => $filePath,
            'foreign_key_constraints' => true,
        ]);

        match ($type) {
            self::TYPE_MASTER => Config::set('tenancy.database.central_connection', 'master'),
            self::TYPE_TENANT => Config::set('database.default', 'tenant'),
        };
    }

    private function disableDatabaseTenancyBootstrapper(): void
    {
        $bootstrappers = array_filter(
            Config::get('tenancy.bootstrappers', []),
            fn ($b) => $b !== DatabaseTenancyBootstrapper::class
        );
        Config::set('tenancy.bootstrappers', $bootstrappers);
    }

    private function registerTenancyConnectionListener(): void
    {
        Event::listen(TenancyInitialized::class, function () {
            if (isset($this->sqliteTestFiles[self::TYPE_TENANT])) {
                $this->configureConnection(
                    self::TYPE_TENANT,
                    $this->sqliteTestFiles[self::TYPE_TENANT]
                );
                DB::purge('tenant');
            }
        });
    }

    private function pointTenantConnectionToMaster(): void
    {
        if (isset($this->sqliteTestFiles[self::TYPE_MASTER])) {
            Config::set('database.connections.tenant', [
                'driver' => 'sqlite',
                'database' => $this->sqliteTestFiles[self::TYPE_MASTER],
                'foreign_key_constraints' => true,
            ]);
        }
    }

    protected function tearDownDatabase(): void
    {
        foreach ($this->sqliteTestFiles as $filePath) {
            if (file_exists($filePath)) {
                @unlink($filePath);
            }
        }
        $this->sqliteTestFiles = [];
    }

    private function getTemplatePath(string $type): string
    {
        return base_path("database/testing/{$type}-template.sqlite");
    }

    private function generateTestPath(string $type): string
    {
        return base_path("database/testing/{$type}-test-" . uniqid('', true) . '.sqlite');
    }

    private function assertTemplateExists(string $path): void
    {
        if (!file_exists($path)) {
            throw new \RuntimeException(
                "Template DB not found at {$path}. Run: php artisan testing:prepare-databases"
            );
        }
    }
}
