<?php

namespace App\Mail;

use Mtc\MercuryDataModels\Enums\VehicleSubscriptionEventType;
use App\Facades\Settings;
use App\Facades\Site;
use App\IntegrationRepository;
use App\Traits\FormatsPhoneNumber;
use App\Traits\GetsColourSettings;
use App\Traits\RetrievesGlobalContentForEmailFooter;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Address;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;
use Mtc\MercuryDataModels\VehicleSubscription;

class VehicleSubscriptionNotificationMail extends Mailable
{
    use Queueable;
    use SerializesModels;
    use GetsColourSettings;
    use FormatsPhoneNumber;
    use RetrievesGlobalContentForEmailFooter;

    public function __construct(
        public readonly VehicleSubscription $subscription,
        public readonly VehicleSubscriptionEventType $eventType,
        public readonly array $eventData = []
    ) {
    }

    public function envelope(): Envelope
    {
        $vehicle = $this->subscription->vehicle;
        $vehicleTitle = $vehicle->title ?? $vehicle->name ?? 'Vehicle';

        $subject = match ($this->eventType) {
            VehicleSubscriptionEventType::PRICE_DROP => __('emails.vehicle_subscription.subject.price_drop', ['vehicle' => $vehicleTitle]),
            VehicleSubscriptionEventType::IMAGES_ADDED => __('emails.vehicle_subscription.subject.images_added', ['vehicle' => $vehicleTitle]),
            VehicleSubscriptionEventType::VIDEO_ADDED => __('emails.vehicle_subscription.subject.video_added', ['vehicle' => $vehicleTitle]),
        };

        return new Envelope(
            from: new Address(config('mail.from.address'), config('mail.from.name')),
            subject: $subject,
        );
    }

    public function content(): Content
    {
        $vehicle = $this->subscription->vehicle;
        $customer = $this->subscription->customer;

        return new Content(
            view: 'emails.vehicle-subscription-notification',
            with: [
                'site_name' => Settings::get('app-name'),
                'site_logo' => Settings::get('site-logo'),
                'site_url' => Site::url(''),
                'site_logo_alternate' => Settings::get('site-alternate-logo'),
                'colours' => $this->getColourSettings(),
                'footer' => $this->getFooterFromGlobalContent('customer-enquiry-autoresponder'),
                'theme' => Settings::get('mail-customer-theme'),
                'phone_number' => Settings::get('app-contact-phone_number'),
                'phone_number_formatted' => $this->getFormattedPhoneNumber(
                    Settings::get('app-contact-phone_number', '')
                ),
                'socials' => app(IntegrationRepository::class)->findNonEmptyValuesForType('socials')
                    ->map(fn($link, $key) => [
                        'key' => $key,
                        'url' => $link,
                    ])->values(),
                'customer' => $customer,
                'vehicle' => $vehicle,
                'vehicle_url' => Site::url('/stock/' . $vehicle->slug),
                'vehicle_image' => $vehicle->primaryMediaUse?->media?->src ?? null,
                'event_type' => $this->eventType,
                'event_type_label' => $this->eventType->label(),
                'event_data' => $this->eventData,
                'unsubscribe_url' => Site::url('/vehicle-alerts/unsubscribe/' . $this->subscription->unsubscribe_token),
                'unsubscribe_all_url' => Site::url('/vehicle-alerts/unsubscribe-all/' . $this->subscription->unsubscribe_token),
            ],
        );
    }
}
