<?php

namespace App\Jobs;

use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\Crm\Traits\RetrievesCustomerId;

class ProcessIncomingEnquiryJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;
    use RetrievesCustomerId;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected EnquiryModel $enquiry)
    {
        $this->onQueue('emails');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        if ($this->process()) {
            $this->enquiry->refresh()->update([
                'ingested_at' => Carbon::now(),
                'customer_id' => $this->getCustomerId($this->enquiry->email, $this->getEnquiryDetails()),
            ]);
        }
    }

    private function process(): bool
    {
        // No confirmation on partial submissions
        if ($this->enquiry->status_id > 0) {
            $this->sendConfirmation();
        }

        $actions = $this->enquiry
            ->actions()
            ->whereNull('processed_at')
            ->where('attempts', '<', config('crm.enquiry_action_max_attempts'))
            ->get();

        if ($actions->isEmpty()) {
            return true;
        }

        // mark as processed if all action was successful
        return $actions->every(fn(EnquiryActionModel $action) => $this->handleEnquiryAction($action));
    }

    private function handleEnquiryAction(EnquiryActionModel $action): bool
    {
        try {
            $handler = Enquiries::makeAction($action->action_name);
            if ($handler->handle($action, $this->enquiry) === false) {
                $action->update([
                    'failed_at' => Carbon::now(),
                    'context' => 'Failed with reason: ' . $handler->failureReason()
                ]);
                $action->increment('attempts');
                return false;
            }

            $action->update([
                'processed_at' => Carbon::now(),
                'failed_at' => null,
                'context' => $handler->successDetails(),
            ]);
        } catch (Exception $exception) {
            Log::error('Enquiry action failed: ' . $exception->getMessage(), [
                'enquiry' => $this->enquiry,
                'action' => $action,
                'error' => $exception,
            ]);
            $action->update([
                'failed_at' => Carbon::now(),
                'context' => 'Failed due to a system error: ' . $exception->getMessage()
            ]);
            $action->increment('attempts');
            return false;
        }

        return true;
    }

    private function sendConfirmation(): void
    {
        if ($this->enquiry->confirmation_sent) {
            return;
        }

        if (empty($this->enquiry->form->send_customer_confirmation_email)) {
            $this->enquiry->refresh()->update([
                'confirmation_sent' => true,
            ]);
            return;
        }

        try {
            $recipient = $this->enquiry->email;
            if (!filter_var($recipient, FILTER_VALIDATE_EMAIL)) {
                return;
            }
            $class = config('crm.enquiry_confirmation_mail');
            Mail::to($this->enquiry->email)->send(new $class($this->enquiry));
            $this->enquiry->refresh()->update([
                'confirmation_sent' => true,
            ]);
        } catch (\Exception $exception) {
            Log::warning($exception->getMessage());
        }
    }

    private function getEnquiryDetails(): array
    {
        $details = [];
        foreach ($this->enquiry->getNormalizedAnswers() as $answer) {
            $is_phone_number = stripos($answer['question'], 'Contact Number') !== false
                || stripos($answer['question'], 'Contact No') !== false
                || stripos($answer['question'], 'Contact Number') !== false;
            if ($is_phone_number) {
                $details['phone_number'] = $answer['answer'];
            }

            $is_first_name = stripos($answer['question'], 'First Name') !== false
                || stripos($answer['question'], 'Firstname') !== false;
            if ($is_first_name) {
                $details['first_name'] = $answer['answer'];
            }

            $is_last_name = stripos($answer['question'], 'Last Name') !== false
                || stripos($answer['question'], 'Lastname') !== false
                || stripos($answer['question'], 'Surname') !== false;
            if ($is_last_name) {
                $details['last_name'] = $answer['answer'];
            }
        }

        return $details;
    }
}
