<?php

namespace App\Crm;

use App\Contracts\AddsDealershipFields;
use App\Contracts\ForwardsReservationToCrm;
use App\Facades\Settings;
use App\Services\DealerWebLmsApi;
use Illuminate\Support\Facades\App;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\VehicleReservations\Reservation;

class SendToDealerWebLms implements EnquiryAction, AddsDealershipFields, ForwardsReservationToCrm
{
    use CheckForExpiredAction;

    public function __construct(private DealerWebLmsApi $api)
    {
        //
    }

    public function enabled(): bool
    {
        return Settings::get('dealerweblms-crm-enabled') === true;
    }

    /**
     * Data fields to add to form
     *
     * @return array
     */
    public function formAdditionalDataFields(): array
    {
        return [];
    }

    /**
     * Data fields to add to dealership
     *
     * @return array
     */
    public function dealershipAdditionalDataFields(): array
    {
        return [
            'dealerweb-dealer-id' => [
                'type' => 'text',
                'label' => 'DealerWeb Dealer ID',
            ],
        ];
    }

    /**
     * Data fields to add to questions
     *
     * @return array[]
     */
    public function formQuestionDataFields(): array
    {
        return [
            'dealer-web-lms-field' => [
                'type' => 'select',
                'label' => 'Field mapped against Dealer Web LMS',
                'choices' => [
                    'DealerNo' => 'Dealer Number (Manufacturer’s dealer number)',
                    'ThirdParty' => 'Third Party Name (Supplier of the lead)',
                    'LeadType' => "Lead type",
                    'Title' => 'Title',
                    'Forename' => 'Forename',
                    'Surname' => 'Surname',
                    'FullName' => 'Full Name (If supplied, split into Forename and Surname)',
                    'Salutation' => 'Salutation',
                    'Company' => 'Company Name',
                    'HouseNameNo' => 'House Name/No.',
                    'Street' => 'Street',
                    'Locale' => 'Suburb/Locale/District',
                    'City' => 'City',
                    'Postcode' => 'Postcode',
                    'Tel_Home' => 'Home Phone Number',
                    'Tel_Mobile' => 'Mobile Phone Number',
                    'Tel_Work' => 'Work Phone Number',
                    'Email' => 'Email Address',
                    'Franchise' => 'Franchise',
                    'Model' => 'Model (The model the customer is enquiring about)',
                    'NewUsed' => 'New or used',
                    'PexMake' => 'Part Exchange Vehicle Make',
                    'PexModel' => 'Part Exchange Vehicle Model',
                    'PexRegNo' => 'Part Exchange Vehicle Registration Number',
                    'PexMileage' => 'Part Exchange Vehicle Mileage',
                    'include_in_details_content' => 'Add to additional lead details',
                    'URL' => 'External URL for additional vehicle details (must start with https://)',
                ],
            ]
        ];
    }

    /**
     * Handle the action
     *
     * @param EnquiryActionModel $action
     * @param EnquiryModel $enquiry
     * @return bool
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        if ($this->expired($action)) {
            return null;
        }

        return $this->api->sendLead($enquiry, $action);
    }

    public function sendReservation(Reservation $reservation): bool
    {
        return $this->api->sendReservation($reservation);
    }

    /**
     * Message when successfully processed
     *
     * @return string
     */
    public function successDetails(): string
    {
        return $this->api->getResponseMessage();
    }

    /**
     * Message when failure detected
     *
     * @return string
     */
    public function failureReason(): string
    {
        return $this->api->getResponseMessage() ?? 'Unknown error';
    }
}
