<?php

namespace Tests\Feature;

use App\Services\StockManagementService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\MercuryDataModels\User;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class StockManagementControllerTest extends TenantTestCase
{
    use UserForTenant;

    private StockManagementService $service;

    protected function setUp(): void
    {
        parent::setUp();
        $this->service = app(StockManagementService::class);
    }

     public function testEndpointWithoutAuth(): void
    {  
       $this->getJson('/api/stock-management/stock-providers?model_type=bar')
            ->assertStatus(401);
    }

    
    public function testCanListAllVehicleModelStockProvider(): void
    {
        $this->createTestVehicles();

        $model_type = 'vehicles';

        $response = $this->actingAs($this->getUser('mtc'))
            ->getJson('/api/stock-management/stock-providers?model_type=' . $model_type);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'model_type' => $model_type,
            ],
        ]);
 
        $providers = $response->json('data.stock_providers');
        $this->assertCount(2, $providers);

        $this->assertContains('vehicle_provider_1', $providers);
        $this->assertContains(StockManagementService::MANUAL_ENTRY_LABEL, $providers);
    }

    public function testCanListAllVehicleOfferModelStockProvider(): void
    {
       $this->createTestVehicleOffers();

        $model_type = 'vehicle_offers';

        $response = $this->actingAs($this->getUser('mtc'))
            ->getJson('/api/stock-management/stock-providers?model_type=' . $model_type);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'model_type' => $model_type,
            ],
        ]);
 
        $providers = $response->json('data.stock_providers');
        $this->assertCount(2, $providers);

        $this->assertContains('offer_provider_1', $providers);
        $this->assertContains(StockManagementService::MANUAL_ENTRY_LABEL, $providers);
    }


    public function testInvalidModelTypeFailsValidation(): void
    {
        $response = $this->actingAs($this->getUser('mtc'))
            ->getJson('/api/stock-management/stock-providers?model_type=invalid');

        $response->assertStatus(422);
        $response->assertJsonValidationErrors(['model_type']);
    }

  
    public function testNullStockProviderFailsValidation(): void
    { 
        $response = $this->actingAs($this->getUser('mtc'))
            ->postJson('/api/stock-management/stock-providers/bulk-unpublish', [
            'stock_provider' => null,
            'model_type' => 'vehicles',
        ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors(['stock_provider']);
    }

    public function testNullModelTypeFailsValidation(): void
    { 
        $response = $this->actingAs($this->getUser('mtc'))
            ->postJson('/api/stock-management/stock-providers/bulk-unpublish', [
            'stock_provider' => 'hello_world',
            'model_type' => null,
        ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors(['model_type']);
    }

   
    public function testCanBulkUnpublishVehiclesFromStockProvider(): void
    {
        $this->createTestVehicles();

        $response = $this->actingAs($this->getUser('mtc'))
            ->postJson('/api/stock-management/stock-providers/bulk-unpublish', [
                'stock_provider' => 'vehicle_provider_1',
                'model_type' => 'vehicles',
            ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'total_unpublished' => 2
            ],
        ]);

        $this->assertEquals(0,Vehicle::query()
            ->where('stock_provider', 'vehicle_provider_1')
            ->where('is_published', 1)
            ->count());
    }

    public function testCanBulkUnpublishVehiclesFromManualEntry(): void
    {
        $this->createTestVehicles();

        $response = $this->actingAs($this->getUser('mtc'))
            ->postJson('/api/stock-management/stock-providers/bulk-unpublish', [
                'stock_provider' => StockManagementService::MANUAL_ENTRY_LABEL,
                'model_type' => 'vehicles',
            ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'total_unpublished' => 1
            ],
        ]);

        $this->assertEquals(0,Vehicle::query()
            ->where('stock_provider', StockManagementService::MANUAL_ENTRY_LABEL)
            ->where('is_published', 1)
            ->count());
    }

      public function testCanBulkUnpublishVehicleOffersFromStockProvider(): void
    {
        $this->createTestVehicleOffers([]);

        $response = $this->actingAs($this->getUser('mtc'))
            ->postJson('/api/stock-management/stock-providers/bulk-unpublish', [
                'stock_provider' => 'offer_provider_1',
                'model_type' => 'vehicle_offers',
            ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'total_unpublished' => 2
            ],
        ]);

        $this->assertEquals(0,Vehicle::query()
            ->where('stock_provider', 'offer_provider_1')
            ->where('is_published', 1)
            ->count());
    }

    public function testCanBulkUnpublishVehicleOffersFromManualEntry(): void
    {
        $this->createTestVehicleOffers([]);

        $response = $this->actingAs($this->getUser('mtc'))
            ->postJson('/api/stock-management/stock-providers/bulk-unpublish', [
                'stock_provider' => StockManagementService::MANUAL_ENTRY_LABEL,
                'model_type' => 'vehicle_offers',
            ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'total_unpublished' => 1
            ],
        ]);

        $this->assertEquals(0,Vehicle::query()
            ->where('stock_provider', StockManagementService::MANUAL_ENTRY_LABEL)
            ->where('is_published', 1)
            ->count());
    }

   
    private function createTestVehicles(): void
    {
        Vehicle::factory()->create([
            'stock_provider' => 'vehicle_provider_1',
            'is_published' => 1,
        ]);
        Vehicle::factory()->create([
            'stock_provider' => 'vehicle_provider_1',
            'is_published' => 1,
        ]);
        Vehicle::factory()->create([
            'stock_provider' => null,
            'is_published' => 1,
        ]); 
    }


    private function createTestVehicleOffers(): void
    {
        VehicleOffer::factory()->create([
            'stock_provider' => 'offer_provider_1',
            'published' => 1,
        ]);
        VehicleOffer::factory()->create([
            'stock_provider' => 'offer_provider_1',
            'published' => 1,
        ]);
        VehicleOffer::factory()->create([
            'stock_provider' => null,
            'published' => 1,
        ]);
    }
   

    
}
