<?php

declare(strict_types=1);

namespace App\Crm;

use App\Contracts\AddsDealershipFields;
use App\Contracts\ForwardsReservationToCrm;
use App\Contracts\ForwardsServiceBookingToCrm;
use App\Facades\Settings;
use App\Services\MotordatCrmApi;
use Illuminate\Support\Facades\App;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\MercuryDataModels\Booking;
use Mtc\VehicleReservations\Reservation;

class SendToMotordat implements
    EnquiryAction,
    AddsDealershipFields,
    ForwardsReservationToCrm,
    ForwardsServiceBookingToCrm
{
    use CheckForExpiredAction;

    private MotordatCrmApi $api;

    public function __construct()
    {
        $this->api = App::make(MotordatCrmApi::class);
    }

    public function enabled(): bool
    {
        return Settings::get('crm-motordat-enabled') === true;
    }

    /**
     * Data fields to add to form
     */
    public function formAdditionalDataFields(): array
    {
        return [];
    }

    /**
     * Data fields to add to questions
     */
    public function formQuestionDataFields(): array
    {
        return [
            'motordat-field' => [
                'type' => 'select',
                'label' => 'Field mapped against Motordat CRM',
                'choices' => [
                    // Customer fields
                    'title' => 'Title',
                    'first_name' => 'First Name',
                    'last_name' => 'Last Name',
                    'tel' => 'Mobile Phone',
                    'home' => 'Home Phone',
                    'office' => 'Office Phone',
                    'email' => 'Email Address',
                    'address1' => 'Address Line 1',
                    'address2' => 'Address Line 2',
                    'address3' => 'Address Line 3',
                    'address4' => 'Address Line 4',
                    'town' => 'Town',
                    'county' => 'County',
                    'postcode' => 'Postcode',
                    'country' => 'Country',
                    'preferredContactMethods' => 'Preferred means of contact',
                    // Enquiry fields
                    //'leadType' => 'Lead Type',
                    'notes' => 'Notes/Comments',
                    //'origin' => 'Lead Origin',
                    'currentMgOwner' => 'Current MG Owner',
                    'currentVehicleModel' => 'Current Vehicle Model',
                    'countryOfPurchase' => 'Country of Purchase',
                    'currentVehicleRegistrationNumber' => 'Current Vehicle Registration',
                    'signup_terms' => 'Signup Terms',
                ],
            ],
        ];
    }

    /**
     * Data fields to add to dealership
     */
    public function dealershipAdditionalDataFields(): array
    {
        return [
            'motordat-crm-dealer-code' => [
                'type' => 'text',
                'label' => 'Motordat CRM Dealer Code',
                'validation' => [
                    'max:10',
                ],
            ],
        ];
    }

    /**
     * Handle the action
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        if ($this->expired($action)) {
            return null;
        }

        return $this->api->sendLead($enquiry, $action);
    }

    /**
     * Send reservation (test drive) to Motordat
     */
    public function sendReservation(Reservation $reservation): bool
    {
        return $this->api->sendReservation($reservation);
    }

    public function sendServiceBooking(Booking $booking): bool
    {
        return $this->api->sendServiceBooking($booking);
    }

    /**
     * Message when successfully processed
     */
    public function successDetails(): string
    {
        return $this->api->getResponseMessage() ?? __('crm.lead_registered');
    }

    /**
     * Message when failed to process
     */
    public function failureReason(): string
    {
        return $this->api->getResponseMessage() ?? 'Unknown error';
    }
}
