<?php

namespace App\Http\Requests;

use App\Services\VehicleBulkUpdateService;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class VehicleBulkUpdateRequest extends FormRequest
{
    private ?array $allowedFields = null;

    public function authorize(): bool
    {
        return Auth::check();
    }

    public function rules(): array
    {
        $allowedFields = $this->getAllowedFields();

        return [
            'mode' => ['required', 'string', Rule::in(['same', 'different'])],
            'vehicle_ids' => ['required', 'array', 'max:100'],
            'vehicle_ids.*' => ['required', 'integer', 'exists:vehicles,id'],

            // Same mode validation
            'field' => [
                'required_if:mode,same',
                'nullable',
                'string',
                Rule::in($allowedFields),
            ],
            'value' => ['required_if:mode,same', 'nullable'],

            // Different mode validation
            'updates' => ['required_if:mode,different', 'nullable', 'array'],
            'updates.*' => ['array'],
            'updates.*.*' => ['nullable'],
        ];
    }

    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            if ($this->input('mode') === 'different') {
                $this->validateDifferentModeUpdates($validator);
            }
        });
    }

    private function validateDifferentModeUpdates($validator): void
    {
        $updates = $this->input('updates', []);
        $vehicleIds = $this->input('vehicle_ids', []);
        $allowedFields = $this->getAllowedFields();

        foreach ($updates as $vehicleId => $fields) {
            if (!in_array((int) $vehicleId, array_map('intval', $vehicleIds))) {
                $validator->errors()->add(
                    "updates.{$vehicleId}",
                    "Vehicle ID {$vehicleId} is not in the selected vehicles list."
                );
                continue;
            }

            if (!is_array($fields)) {
                continue;
            }

            foreach (array_keys($fields) as $field) {
                if (!in_array($field, $allowedFields)) {
                    $validator->errors()->add(
                        "updates.{$vehicleId}.{$field}",
                        "Field '{$field}' is not allowed for bulk update."
                    );
                }
            }
        }
    }

    private function getAllowedFields(): array
    {
        if ($this->allowedFields === null) {
            $service = app(VehicleBulkUpdateService::class);
            $this->allowedFields = $service->getAllowedFields();
        }

        return $this->allowedFields;
    }

    public function messages(): array
    {
        return [
            'mode.required' => 'Please select an update mode.',
            'mode.in' => 'Invalid update mode. Must be "same" or "different".',
            'vehicle_ids.required' => 'Please select at least one vehicle.',
            'vehicle_ids.max' => 'You can only update up to 100 vehicles at a time.',
            'vehicle_ids.*.exists' => 'One or more selected vehicles do not exist.',
            'field.required_if' => 'Please select a field to update.',
            'field.in' => 'The selected field is not allowed for bulk update.',
            'value.required_if' => 'Please provide a value for the selected field.',
            'updates.required_if' => 'Please provide updates for the selected vehicles.',
        ];
    }
}
