<?php

namespace Tests\Unit;

use App\Facades\Settings;
use App\Models\ImportCondition;
use App\Traits\ImportChecksConditions;
use Carbon\Carbon;
use Illuminate\Support\Facades\Queue;
use Mtc\ContentManager\Models\Media;
use Mtc\MercuryDataModels\Vehicle;
use Tests\TenantTestCase;

class VehicleImportConditionTest extends TenantTestCase
{
    use ImportChecksConditions;

    public function testNoImportConditions()
    {
        $vehicle = Vehicle::factory()->create(['is_published' => true]);
        $vehicle2 = Vehicle::factory()->create(['is_published' => false]);

        $this->assertTrue($this->shouldBePublished(fn() => $vehicle->is_published, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => $vehicle2->is_published, 'test', $vehicle2));
    }

    public function testImportConditionNoRules()
    {
        $vehicle = Vehicle::factory()->create(['is_published' => true]);
        $vehicle2 = Vehicle::factory()->create(['is_published' => false]);
        ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => $vehicle->is_published, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => $vehicle2->is_published, 'test', $vehicle2));
    }

    public function testImportConditionSingleRule()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 2]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => '=',
            'value' => 2
        ]);

        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
    }
    public function testImportConditionMultipleRule()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1, 'fuel_type_id' => 3]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 1, 'fuel_type_id' => 1]);
        $vehicle3 = Vehicle::factory()->create(['make_id' => 2, 'fuel_type_id' => 3]);
        $vehicle4 = Vehicle::factory()->create(['make_id' => 3, 'fuel_type_id' => 1]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => '=',
            'value' => 2
        ]);

        $condition->rules()->create([
            'field' => 'fuel_type_id',
            'condition' => '=',
            'value' => 3
        ]);

        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle3));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle4));
    }

    public function testMultipleImportConditionsWithOneMatch()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1, 'fuel_type_id' => 3]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 1, 'fuel_type_id' => 1]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => '=',
            'value' => 2
        ]);
        $condition2 = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition2->rules()->create([
            'field' => 'fuel_type_id',
            'condition' => '=',
            'value' => 3
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
    }

    public function testMultipleImportConditionsAllMatch()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1, 'fuel_type_id' => 3]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 1, 'fuel_type_id' => 1]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => '=',
            'value' => 1
        ]);
        $condition2 = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition2->rules()->create([
            'field' => 'fuel_type_id',
            'condition' => '=',
            'value' => 3
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => $vehicle->is_published, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => $vehicle2->is_published, 'test', $vehicle2));
    }

    public function testNotEquals()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 2]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => '!=',
            'value' => 2
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
    }

    public function testGreaterThan()
    {
        $vehicle = Vehicle::factory()->create(['price' => 10000]);
        $vehicle2 = Vehicle::factory()->create(['price' => 12000]);
        $vehicle3 = Vehicle::factory()->create(['price' => 9000]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'price',
            'condition' => '>',
            'value' => 10000
        ]);

        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testGreaterThanEquals()
    {
        $vehicle = Vehicle::factory()->create(['price' => 10000]);
        $vehicle2 = Vehicle::factory()->create(['price' => 12000]);
        $vehicle3 = Vehicle::factory()->create(['price' => 9000]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'price',
            'condition' => '>=',
            'value' => 10000
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testLessThan()
    {
        $vehicle = Vehicle::factory()->create(['price' => 10000]);
        $vehicle2 = Vehicle::factory()->create(['price' => 12000]);
        $vehicle3 = Vehicle::factory()->create(['price' => 9000]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'price',
            'condition' => '<',
            'value' => 10000
        ]);

        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testLessThanEquals()
    {
        $vehicle = Vehicle::factory()->create(['price' => 10000]);
        $vehicle2 = Vehicle::factory()->create(['price' => 12000]);
        $vehicle3 = Vehicle::factory()->create(['price' => 9000]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'price',
            'condition' => '<=',
            'value' => 10000
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testEmpty()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => null]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 1]);
        $vehicle3 = Vehicle::factory()->create(['make_id' => 2]);
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'is_null',
            'value' => '',
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testImageChecksumMatch()
    {
        Queue::fake();
        $m1 = Media::factory()->create(['source_filename' => 'foo.jpg']);
        $m2 = Media::factory()->create(['source_filename' => 'faz.jpg']);
        $m3 = Media::factory()->create(['source_filename' => 'baz.jpg']);

        $cheksum = dechex(crc32(json_encode([3, 'foo.jpg', 'faz.jpg', 'baz.jpg'])));

        $v1 = Vehicle::factory()->create();
        $v2 = Vehicle::factory()->create(['image_checksum' => 'foo']);
        $v3 = Vehicle::factory()->create(['image_checksum' => $cheksum]);

        $images = ['foo.jpg', 'faz.jpg', 'baz.jpg'];

        \Mtc\ContentManager\Facades\Media::setUsesForModel([$m1->id, $m2->id, $m3->id], $v3);
        $this->assertFalse($this->imageChecksumMismatch($images, $v3));

        \Mtc\ContentManager\Facades\Media::setUsesForModel([$m1->id, $m2->id, $m3->id], $v1);
        $this->assertFalse($this->imageChecksumMismatch($images, $v1));

        $images = ['foo.jpg', 'faz.jpg', 'baz.jpg', 'bar.jpg'];
        $this->assertTrue($this->imageChecksumMismatch($images, $v2));
        $this->assertTrue($this->imageChecksumMismatch($images, $v3));
        $images = ['foo.jpg', 'faz.jpg', 'bar.jpg'];
        $this->assertTrue($this->imageChecksumMismatch($images, $v2));
        $this->assertTrue($this->imageChecksumMismatch($images, $v3));
    }

    // ========================================
    // Tests for "in" condition
    // ========================================

    public function testInConditionWithSingleValue()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 2]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'in',
            'value' => '1'
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
    }

    public function testInConditionWithMultipleValues()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 2]);
        $vehicle3 = Vehicle::factory()->create(['make_id' => 3]);
        $vehicle4 = Vehicle::factory()->create(['make_id' => 4]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'in',
            'value' => '[1,2,3]'
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle3));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle4));
    }

    public function testNotInCondition()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 2]);
        $vehicle3 = Vehicle::factory()->create(['make_id' => 3]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'not_in',
            'value' => '[1,2]'
        ]);

        // Vehicles with make_id 1 and 2 should be published (not in the exclusion list)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        // Vehicle with make_id 3 should NOT be published (it's not in 1,2, so condition matches)
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testInConditionWithSpacesInValues()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 2]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'in',
            'value' => '[1, 2, 3]' // With spaces in JSON
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
    }

    // ========================================
    // Tests for "age" calculated field
    // ========================================

    public function testAgeFieldInYears()
    {
        // Vehicle registered 3 years ago
        $vehicle = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(3),
        ]);
        // Vehicle registered 5 years ago
        $vehicle2 = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(5),
        ]);
        // Vehicle registered 2 years ago
        $vehicle3 = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(2),
        ]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'age',
            'condition' => '>',
            'value' => 3
        ]);

        // Vehicle with age 3 should be published (not > 3)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        // Vehicle with age 5 should NOT be published (> 3)
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        // Vehicle with age 2 should be published (not > 3)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testAgeFieldEquals()
    {
        $vehicle = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(5),
        ]);
        $vehicle2 = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(3),
        ]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'age',
            'condition' => '=',
            'value' => 5
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
    }

    public function testAgeFieldWithNullDate()
    {
        $vehicle = Vehicle::factory()->create([
            'first_registration_date' => null,
        ]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'age',
            'condition' => 'is_null',
            'value' => ''
        ]);

        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
    }

    // ========================================
    // Tests for "age_months" calculated field
    // ========================================

    public function testAgeMonthsField()
    {
        // Vehicle registered 18 months ago
        $vehicle = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subMonths(18),
        ]);
        // Vehicle registered 30 months ago
        $vehicle2 = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subMonths(30),
        ]);
        // Vehicle registered 12 months ago
        $vehicle3 = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subMonths(12),
        ]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'age_months',
            'condition' => '>=',
            'value' => 24
        ]);

        // Vehicle with 18 months should be published (< 24)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        // Vehicle with 30 months should NOT be published (>= 24)
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        // Vehicle with 12 months should be published (< 24)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    // ========================================
    // Tests for "odometer" calculated field
    // ========================================

    public function testOdometerFieldWithMilesSetting()
    {
        // Create vehicles before mocking to avoid mock intercepting all Settings calls
        $vehicle = Vehicle::factory()->create([
            'odometer_mi' => 50000,
            'odometer_km' => 80467,
        ]);
        $vehicle2 = Vehicle::factory()->create([
            'odometer_mi' => 100000,
            'odometer_km' => 160934,
        ]);
        $vehicle3 = Vehicle::factory()->create([
            'odometer_mi' => 30000,
            'odometer_km' => 48280,
        ]);

        Settings::partialMock()
            ->shouldReceive('get')
            ->with('automotive-distance_measurement', 'mi')
            ->andReturn('mi');

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'odometer',
            'condition' => '>',
            'value' => 75000 // 75,000 miles
        ]);

        // 50k miles - should be published
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        // 100k miles - should NOT be published
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        // 30k miles - should be published
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testOdometerFieldWithKilometersSetting()
    {
        // Create vehicles before mocking to avoid mock intercepting all Settings calls
        $vehicle = Vehicle::factory()->create([
            'odometer_mi' => 50000,
            'odometer_km' => 80000,
        ]);
        $vehicle2 = Vehicle::factory()->create([
            'odometer_mi' => 100000,
            'odometer_km' => 160000,
        ]);

        Settings::partialMock()
            ->shouldReceive('get')
            ->with('automotive-distance_measurement', 'mi')
            ->andReturn('km');

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'odometer',
            'condition' => '>',
            'value' => 100000 // 100,000 km
        ]);

        // 80k km - should be published
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle));
        // 160k km - should NOT be published
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle2));
    }

    // ========================================
    // Tests for skipImporting with new fields
    // ========================================

    public function testSkipImportingWithAgeField()
    {
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_import' => true,
        ]);
        $condition->rules()->create([
            'field' => 'age',
            'condition' => '>',
            'value' => 5
        ]);

        // 3 years old - should not skip
        $data1 = ['first_registration_date' => Carbon::now()->subYears(3)->format('Y-m-d')];
        $this->assertFalse($this->skipImporting('test', $data1));

        // 7 years old - should skip
        $data2 = ['first_registration_date' => Carbon::now()->subYears(7)->format('Y-m-d')];
        $this->assertTrue($this->skipImporting('test', $data2));
    }

    public function testSkipImportingWithOdometerField()
    {
        Settings::shouldReceive('get')
            ->with('automotive-distance_measurement', 'mi')
            ->andReturn('mi');

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_import' => true,
        ]);
        $condition->rules()->create([
            'field' => 'odometer',
            'condition' => '>',
            'value' => 100000
        ]);

        // 50k miles - should not skip
        $data1 = ['odometer_mi' => 50000];
        $this->assertFalse($this->skipImporting('test', $data1));

        // 150k miles - should skip
        $data2 = ['odometer_mi' => 150000];
        $this->assertTrue($this->skipImporting('test', $data2));
    }

    public function testSkipImportingWithInCondition()
    {
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_import' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'in',
            'value' => '[1,2,3]'
        ]);

        // make_id 1 - should skip
        $this->assertTrue($this->skipImporting('test', ['make_id' => 1]));
        // make_id 2 - should skip
        $this->assertTrue($this->skipImporting('test', ['make_id' => 2]));
        // make_id 4 - should not skip
        $this->assertFalse($this->skipImporting('test', ['make_id' => 4]));
    }

    public function testSkipImportingWithNotInCondition()
    {
        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_import' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'not_in',
            'value' => '[1,2]'
        ]);

        // make_id 1 - should not skip (it's in the allowed list)
        $this->assertFalse($this->skipImporting('test', ['make_id' => 1]));
        // make_id 3 - should skip (not in 1,2)
        $this->assertTrue($this->skipImporting('test', ['make_id' => 3]));
    }

    // ========================================
    // Tests for combined conditions
    // ========================================

    public function testCombinedAgeAndOdometerConditions()
    {
        // Create vehicles before mocking to avoid mock intercepting all Settings calls
        // Vehicle older than 5 years with high mileage
        $vehicle = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(7),
            'odometer_mi' => 120000,
        ]);
        // Vehicle older than 5 years but low mileage
        $vehicle2 = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(7),
            'odometer_mi' => 50000,
        ]);
        // New vehicle with high mileage
        $vehicle3 = Vehicle::factory()->create([
            'first_registration_date' => Carbon::now()->subYears(2),
            'odometer_mi' => 120000,
        ]);

        Settings::partialMock()
            ->shouldReceive('get')
            ->with('automotive-distance_measurement', 'mi')
            ->andReturn('mi');

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        // Both rules must match (AND logic)
        $condition->rules()->create([
            'field' => 'age',
            'condition' => '>',
            'value' => 5
        ]);
        $condition->rules()->create([
            'field' => 'odometer',
            'condition' => '>',
            'value' => 100000
        ]);

        // Old + high mileage - should NOT be published
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        // Old + low mileage - should be published (only age matches)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        // New + high mileage - should be published (only mileage matches)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    public function testInConditionWithMakeAndOtherRules()
    {
        $vehicle = Vehicle::factory()->create(['make_id' => 1, 'price' => 5000]);
        $vehicle2 = Vehicle::factory()->create(['make_id' => 1, 'price' => 15000]);
        $vehicle3 = Vehicle::factory()->create(['make_id' => 5, 'price' => 5000]);

        $condition = ImportCondition::query()->create([
            'provider' => 'test',
            'active' => true,
            'do_not_publish' => true,
        ]);
        $condition->rules()->create([
            'field' => 'make_id',
            'condition' => 'in',
            'value' => '[1,2,3]'
        ]);
        $condition->rules()->create([
            'field' => 'price',
            'condition' => '<',
            'value' => 10000
        ]);

        // make_id 1 + price 5000 - should NOT be published (both match)
        $this->assertFalse($this->shouldBePublished(fn() => true, 'test', $vehicle));
        // make_id 1 + price 15000 - should be published (only make matches)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle2));
        // make_id 5 + price 5000 - should be published (only price matches)
        $this->assertTrue($this->shouldBePublished(fn() => true, 'test', $vehicle3));
    }

    /**
     * Required by ImportChecksConditions trait
     */
    public function getProviderName(): string
    {
        return 'test';
    }
}
