<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreFranchiseRequest;
use App\Http\Requests\UpdateFranchiseRequest;
use App\Http\Resources\FranchiseList;
use App\Http\Resources\FranchiseUsageListResource;
use App\Http\Resources\FranchiseView;
use Illuminate\Http\Request;
use Mtc\ContentManager\Facades\Media;
use Mtc\MercuryDataModels\Franchise;

class FranchiseController extends Controller
{
    public function __construct()
    {
        $this->middleware([
            'permission:edit-franchises',
        ]);
    }

    public function index(Request $request)
    {
        $perPage = min((int) $request->input('per_page', 15), 200);

        $franchises = Franchise::query()
            ->with('make')
            ->setSortBy($request->input('sort_by', 'id_desc'))
            ->setFilters($request->input('filters', []))
            ->when(
                $request->filled('search_term'),
                function ($query) use ($request) {
                    $query->where('name', 'like', '%' . $request->input('search_term') . '%');
                }
            )
            ->paginate($perPage);

        return new FranchiseList($franchises);
    }

    public function show(Franchise $franchise)
    {
        return new FranchiseView($franchise);
    }

    public function store(StoreFranchiseRequest $request)
    {
        return Franchise::query()->create($request->input());
    }

    public function update(UpdateFranchiseRequest $request, Franchise $franchise)
    {
        $franchise->fill($request->input());
        $franchise->slug = $request->input('seo.slug') ?? $franchise->slug;
        $franchise->save();
        Media::setUsesForModel($request->input('media', []), $franchise, ['primary' =>  true], false);
        Media::setUsesForModel($request->input('square_logo', []), $franchise, ['flags' => ['square' => true]], false);
        $allMedia = array_merge(
            $request->input('media', []),
            $request->input('square_logo', [])
        );
        Media::removeMediaUsesNotInList($franchise, $allMedia);

        return new FranchiseView($franchise);
    }

    public function destroy(Franchise $franchise)
    {
        return $franchise->delete();
    }

    public function copy(Request $request, Franchise $franchise): FranchiseView
    {
        $data = $franchise->toArray();
        $data['name'] = $request->input('name');
        unset($data['slug']);
        $copy = Franchise::query()->create($data);
        return new FranchiseView($copy);
    }

    public function usage(Franchise $franchise)
    {
        $entries = $franchise->pages()->get();
        $franchise->menus()->get()->each(fn($menu) => $entries->push($menu));
        $franchise->newCars()->get()->each(fn($car) => $entries->push($car));
        $franchise->dealerships()->get()->each(fn($dealership) => $entries->push($dealership));
        $franchise->offers()->get()->each(fn($offer) => $entries->push($offer));
        $franchise->labels()->get()->each(fn($label) => $entries->push($label));

        return new FranchiseUsageListResource($entries);
    }
}
