<?php

namespace App\Exports;

use App\Facades\Site;
use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Vehicle;

class MotorsCoUkExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'bodyStyle',
                'fuelType',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::MOTORS_CO_UK),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::MOTORS_CO_UK)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'Feed_ID' => $vehicle->dealership?->data['motors-co-uk-dealer-id'] ?? null,
                    'Vehicle_ID' => $vehicle->uuid,
                    'Make' => $vehicle->make?->name,
                    'Model' => $vehicle->model?->name,
                    'Variant' => $vehicle->derivative,
                    'EngineSize' => $vehicle->engine_size_cc,
                    'FuelType' => $vehicle->fuelType?->name,
                    'TransmissionType' => $vehicle->transmission_type,
                    'VehicleType' => $vehicle->type === VehicleType::CAR ? 'CARS' : 'COMM',
                    'RegistrationDate' => $vehicle->first_registration_date,
                    'Registration' => str_replace(' ', '', $vehicle->registration_number),
                    'Vin' => $vehicle->vin,
                    'NoDoors' => $vehicle->door_count,
                    'BodyStyle' => $vehicle->bodyStyle?->name,
                    'Colour' => $vehicle->colour,
                    'Mileage' => $vehicle->odometer_mi,
                    'Price' => $vehicle->price,
                    'PricePlusVat' => $vehicle->type === VehicleType::CAR ? 0 : 1,
                    'New' => $vehicle->is_new,
                    'Description' => $vehicle->specs
                        ->pluck('description')
                        ->implode(','),
                    'PictureRefs' => $vehicle->mediaUses
                        ->filter(fn($mediaUse) => $mediaUse->media)
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large'))
                        ->implode(','),
                    'ProductURL' => Site::vehicleUrl($vehicle),
                    'Cap_ID' => $vehicle->cap_id,
                    'SalesTagline' => $vehicle->attention_grabber
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'Feed_ID',
            'Vehicle_ID',
            'Make',
            'Model',
            'Variant',
            'EngineSize',
            'FuelType',
            'TransmissionType',
            'VehicleType',
            'RegistrationDate',
            'Registration',
            'Vin',
            'NoDoors',
            'BodyStyle',
            'Colour',
            'Mileage',
            'Price',
            'PricePlusVat',
            'New',
            'Description',
            'PictureRefs',
            'ProductURL',
            'Cap_ID',
            'SalesTagline'
        ];
    }
}
