<?php

namespace Tests\Tenant\Controllers;

use Mtc\MercuryDataModels\Property;
use Mtc\MercuryDataModels\PropertyCategory;
use Mtc\MercuryDataModels\PropertyCategoryField;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class PropertyControllerTest extends TenantTestCase
{
    use UserForTenant;

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        $category = PropertyCategory::factory()->create();
        Property::factory(5)->create(['category_id' => $category->id]);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.properties.index', $category->slug));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertCount(5, $response->json('data'));
    }

    public function testStore()
    {
        $category = PropertyCategory::factory()->create();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.properties.store', $category->slug), [
                'name' => 'foo',
            ]);

        $response->assertStatus(201);
        $this->assertEquals('foo', $response->json('name'));
        $this->assertTrue(Property::query()->where('category_id', $category->id)->where('name', 'foo')->exists());

    }

    public function testShow()
    {
        $category = PropertyCategory::factory()->create();
        $property = Property::factory()->create([
            'name' => 'foo',
            'category_id' => $category->id,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.properties.show', [$category->slug, $property]));

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('name'));
    }

    public function testUpdate()
    {
        $category = PropertyCategory::factory()->create();
        $field = PropertyCategoryField::factory()->create([
            'name' => 'foo',
            'type' => 'text',
        ]);

        $property = Property::factory()->create([
            'name' => 'foo',
            'category_id' => $category->id,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.properties.update', [$category->slug, $property]), [
                'name' => 'baz',
                'slug' => 'foo',
                'active' => true,
                'fields' => [
                    [
                        'id' => $field->id,
                        'slug' => $field->slug,
                        'type' => $field->type,
                        'value' => 'foo baz bar',
                    ]
                ]
            ]);

        $property->refresh();
        $response->assertStatus(200);
        $this->assertEquals('baz', $response->json('name'));
        $this->assertTrue($response->json('active'));
        $value = $property->attributeValues()->first();
        $this->assertEquals($field->slug, $value->slug);
        $this->assertEquals($field->id, $value->field_id);
        $this->assertEquals('foo baz bar', $value->getValue());

    }

    public function testDestroy()
    {
        $category = PropertyCategory::factory()->create();
        $property = Property::factory()->create([
            'name' => 'foo',
            'category_id' => $category->id,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->deleteJson(route('tenant.properties.destroy', [$category->slug, $property]));

        $response->assertStatus(200);
        $this->assertFalse(Property::query()->where('id', $property->id)->exists());
    }

    public function testReorder()
    {
        $category = PropertyCategory::factory()->create();
        $property = Property::factory()->create([
            'name' => 'foo',
            'category_id' => $category->id,
            'order' => 3,
        ]);
        $property2 = Property::factory()->create([
            'name' => 'foo',
            'category_id' => $category->id,
            'order' => 2,
        ]);
        $property3 = Property::factory()->create([
            'name' => 'foo',
            'category_id' => $category->id,
            'order' => 1,
        ]);

        $this->assertEquals(3, $property->order);
        $this->assertEquals(2, $property2->order);
        $this->assertEquals(1, $property3->order);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.properties.reorder', [$category->slug]), [
                'properties' => [
                    $property,
                    $property2,
                    $property3,
                ]
            ]);

        $response->assertStatus(200);
        $property->refresh();
        $property2->refresh();
        $property3->refresh();
        $this->assertEquals(0, $property->order);
        $this->assertEquals(1, $property2->order);
        $this->assertEquals(2, $property3->order);

    }
}
