<?php

namespace Tests\Unit\Services;

use App\Facades\Settings;
use App\Services\RemoveBg;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Tests\TestCase;
use Mockery;

class RemoveBgTest extends TestCase
{
    public function testHandleDoesNothingWhenFeatureIsDisabled()
    {
        Settings::shouldReceive('get')
            ->with('remove-bg-enabled')
            ->once()
            ->andReturn(false);

        $file = Mockery::mock(UploadedFile::class);

        Http::fake();

        (new RemoveBg())->handle($file);

        Http::assertNothingSent();
    }

    public function testHandleSkipsWhenApiKeyIsMissing()
    {
        Settings::shouldReceive('get')
            ->with('remove-bg-enabled')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('remove-bg-api_key')
            ->once()
            ->andReturn('');

        $file = Mockery::mock(UploadedFile::class);

        Http::fake();

        (new RemoveBg())->handle($file);

        Http::assertNothingSent();
    }

    public function testHandleLogsErrorOnApiFailure()
    {
        Settings::shouldReceive('get')
            ->with('remove-bg-enabled')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('remove-bg-api_key')
            ->twice()
            ->andReturn('test-api-key');

        $file = Mockery::mock(UploadedFile::class);
        $file->shouldReceive('getPathname')->once()->andReturn('/tmp/image.jpg');
        $file->shouldReceive('getClientOriginalName')->once()->andReturn('image.jpg');

        Http::fake([
            'https://api.remove.bg/*' => Http::response('API error', 400)
        ]);

        Log::shouldReceive('error')
            ->once()
            ->with('Failed to remove background. Error: API error');

        // Simulate image file read
        file_put_contents('/tmp/image.jpg', 'dummy content');

        (new RemoveBg())->handle($file);
    }

    public function testHandleOverwritesImageOnSuccess()
    {
        Settings::shouldReceive('get')
            ->with('remove-bg-enabled')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('remove-bg-api_key')
            ->twice()
            ->andReturn('test-api-key');

        $originalPath = '/tmp/test-image.jpg';
        file_put_contents($originalPath, 'original content');

        $file = Mockery::mock(UploadedFile::class);
        $file->shouldReceive('getPathname')->twice()->andReturn($originalPath);
        $file->shouldReceive('getClientOriginalName')->once()->andReturn('image.jpg');

        Http::fake([
            'https://api.remove.bg/*' => Http::response('processed image content', 200)
        ]);

        (new RemoveBg())->handle($file);

        $this->assertSame('processed image content', file_get_contents($originalPath));
    }

    public function testEnabledReturnsTrueOnlyWhenEnabledAndApiKeyExists()
    {
        Settings::shouldReceive('get')
            ->with('remove-bg-enabled')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('remove-bg-api_key')
            ->once()
            ->andReturn('some-key');

        $this->assertTrue(RemoveBg::enabled());
    }

    public function testEnabledReturnsFalseWhenDisabled()
    {
        Settings::shouldReceive('get')
            ->with('remove-bg-enabled')
            ->once()
            ->andReturn(false);

        $this->assertFalse(RemoveBg::enabled());
    }

    public function testEnabledReturnsFalseWhenApiKeyIsMissing()
    {
        Settings::shouldReceive('get')
            ->with('remove-bg-enabled')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('remove-bg-api_key')
            ->once()
            ->andReturn('');

        $this->assertFalse(RemoveBg::enabled());
    }

    protected function tearDown(): void
    {
        parent::tearDown();
        Mockery::close();
    }
}
