<?php

namespace App\Http\Resources;

use App\Events\LoadDealershipDataFields;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\MercuryDataModels\Dealership;

class DealershipListResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $data_fields = $this->loadDataFields();
        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource
                ->through(fn(Dealership $dealership) => $this->mapLocation($dealership, $data_fields));
        } else {
            $this->resource = $this->resource
                ->map(fn(Dealership $dealership) => $this->mapLocation($dealership, $data_fields));
        }
        $resource = $this->resource->toArray();
        $resource['default_columns'] = [
            'vehicle_count',
            'address',
            'contact_no',
        ];

        $resource['view_columns'] = array_merge([
            [
                'id' => 'vehicle_count',
                'name' => 'Vehicle count',
                'icon' => 'cars',
            ],
            [
                'id' => 'address',
                'name' => 'Address',
                'icon' => 'location-dot',
            ],
            [
                'id' => 'contact_no',
                'name' => 'Phone number',
                'icon' => 'phone',
            ],
        ], $data_fields->values()->toArray());
        return $resource;
    }

    /**
     * Convert vehicle to the json format used in lists
     *
     * @param Dealership $dealership
     * @return array
     */
    protected function mapLocation(Dealership $dealership, $data_fields)
    {
        $extra_fields = $data_fields->map(fn($field) => $dealership->data[$field['id']] ?? null)->toArray();
        return [
            'image' => [
                'src' => $dealership->getPreviewImage('mini-thumb'),
                'alt' => $dealership->name,
            ],
            'active' => $dealership->active,
            'title' => $dealership->name,
            'metaItems' => array_merge([
                'vehicle_count' => $dealership->vehicle_count,
                'address' => collect([
                    $dealership->address1,
                    $dealership->address2,
                    $dealership->city,
                    $dealership->postcode,
                ])->filter()->implode(', '),
                'contact_no' => $dealership->contact_no,
            ], $extra_fields),
            'id' => $dealership->id,
        ];
    }

    private function loadDataFields()
    {
        return collect(event(new LoadDealershipDataFields()))
            ->filter()
            ->mapWithKeys(fn($field) => $field)
            ->map(function ($field, $key) {
                $field['id'] = $key;
                $field['name'] = $field['label'];
                $field['icon'] = 'code-branch';
                return $field;
            });
    }
}
