<?php

namespace Tests\Tenant;

use App\Jobs\ProcessIncomingEnquiryJob;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Models\EnquiryAction;
use Mtc\Crm\Models\FormAction;
use Mtc\Crm\Models\FormSection;
use Mtc\MercuryDataModels\Enquiry;
use Mtc\MercuryDataModels\Form;
use Tests\TenantTestCase;

class ProcessIncomingEnquiryJobTest extends TenantTestCase
{
    public function testProcessesEnquiryAndSetsIngestedAt()
    {
        Mail::fake();

        $form = Form::factory()->create(['send_customer_confirmation_email' => false]);
        $enquiry = Enquiry::factory()->create([
            'form_id' => $form->id,
            'status_id' => 1,
            'email' => 'test@example.com',
            'ingested_at' => null,
        ]);

        (new ProcessIncomingEnquiryJob($enquiry))->handle();

        $enquiry->refresh();
        $this->assertNotNull($enquiry->ingested_at);
    }

    public function testSkipsPartialSubmissions()
    {
        Mail::fake();

        $form = Form::factory()->create(['send_customer_confirmation_email' => false]);
        $enquiry = Enquiry::factory()->create([
            'form_id' => $form->id,
            'status_id' => 0,
            'email' => 'test@example.com',
            'ingested_at' => null,
            'confirmation_sent' => false,
        ]);

        (new ProcessIncomingEnquiryJob($enquiry))->handle();

        // Should still process but not send confirmation for partial submissions
        $enquiry->refresh();
        $this->assertNotNull($enquiry->ingested_at);
    }

    public function testProcessesEnquiryActions()
    {
        Mail::fake();

        $form = Form::factory()->create(['send_customer_confirmation_email' => false]);
        FormSection::factory()->create(['form_id' => $form->id]);

        $enquiry = Enquiry::factory()->create([
            'form_id' => $form->id,
            'status_id' => 1,
            'email' => 'test@example.com',
            'ingested_at' => null,
        ]);

        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'test-action',
            'processed_at' => null,
            'attempts' => 0,
        ]);

        // Action won't be found by Enquiries::makeAction, but job should handle gracefully
        (new ProcessIncomingEnquiryJob($enquiry))->handle();

        $enquiry->refresh();
        // Enquiry should still exist
        $this->assertNotNull($enquiry);
    }

    public function testSetsCustomerIdFromEnquiryDetails()
    {
        Mail::fake();

        $form = Form::factory()->create(['send_customer_confirmation_email' => false]);
        $enquiry = Enquiry::factory()->create([
            'form_id' => $form->id,
            'status_id' => 1,
            'email' => 'customer@example.com',
            'ingested_at' => null,
            'customer_id' => null,
            'details' => [
                '1' => [
                    'question' => 'First Name',
                    'answer' => 'John',
                ],
                '2' => [
                    'question' => 'Last Name',
                    'answer' => 'Doe',
                ],
                '3' => [
                    'question' => 'Contact Number',
                    'answer' => '07123456789',
                ],
            ],
        ]);

        (new ProcessIncomingEnquiryJob($enquiry))->handle();

        $enquiry->refresh();
        $this->assertNotNull($enquiry->ingested_at);
    }
}
