<?php

namespace App\Exports;

use App\Facades\Settings;
use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class AutowebExport implements FromCollection
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::AUTOWEB),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::AUTOWEB)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'vehicle' => [
                        'type' => $vehicle->type == VehicleType::LCV->value ? 'Commercial' : 'Car',
                        'model' => $vehicle->model?->name,
                        'make' => $vehicle->make?->name,
                        'price' => round($vehicle->price),
                        'registration' => str_replace(' ', '', $vehicle->registration_number),
                        'externalVehicleId' => $vehicle->uuid,
                        'variant' => $vehicle->derivative,
                        'mileage' => $vehicle->odometer_mi,
                        'engine' => $vehicle->engine_size_cc,
                        'gearbox' => $vehicle->transmission?->name,
                        'fueltype' => $vehicle->fuelType?->name,
                        'year' => $vehicle->manufacture_year,
                        'colour' => $vehicle->colour,
                        'doors' => $vehicle->door_count,
                        'bodystyle' => $vehicle->bodyStyle?->name,
                        'spec' => $vehicle->specs
                            ->pluck('description')
                            ->toArray(),
                        'description' => $vehicle->description,
                        'registrationDate' => $vehicle->first_registration_date?->format('d/m/Y'),
                        'sold' => $vehicle->is_sold ? 'Sold' : null,
                        'cap_id' => $vehicle->cap_id,
                        'status' => $vehicle->is_reserved ? 'Reserved' : null,
                        'isNew' => $vehicle->is_new ? 1 : 0,

                    ],
                    'media' => [
                        'images' => $vehicle->mediaUses
                            ->map(fn(MediaUse $mediaUse) => $mediaUse->media->getOriginalUrlAttribute(true))
                            ->toArray(),
                        'video' => !empty($vehicle->main_video_url)
                            ? [$vehicle->main_video_url] :
                            []
                    ]
                ];
            });
    }
}
