<?php

namespace App\Exports;

use App\Facades\Settings;
use App\Facades\Site;
use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class MotoNovoExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'bodyStyle',
                'fuelType',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::MOTO_NOVO),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::MOTO_NOVO)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'Feed_Id' => $vehicle->dealership?->data['moto-novo-dealer-id'],
                    'Vehicle_ID' => $vehicle->uuid,
                    'VRM' => $vehicle->registration_number,
                    'VRMYear' => $vehicle->manufacture_year,
                    'VIN' => $vehicle->vin,
                    'Date First Registered' => $vehicle->first_registration_date,
                    'MOT Expiry Date' => '',
                    'Make' => $vehicle->make?->name,
                    'Model' => $vehicle->model?->model,
                    'Variant' => $vehicle->derivative,
                    'VehicleType' => $vehicle->type,
                    'Mileage' => $vehicle->odometer_mi,
                    'BodyType' => $vehicle->bodyStyle?->name,
                    'Doors' => $vehicle->door_count,
                    'Colour' => $vehicle->colour,
                    'FuelType' => $vehicle->fuelType?->name,
                    'Engine Size' => $vehicle->engine_size_cc,
                    'Power Output' => '',
                    'CO2 (g/km)' => $vehicle->co2,
                    'Transmission' => $vehicle->transmission?->name,
                    'Previous Owners' => $vehicle->previous_owner_count,
                    'Price' => $vehicle->price,
                    'Price Includes Vat' => ($vehicle->type === VehicleType::LCV) ? 'false' : 'true',
                    'TradePrice' => null,
                    'TradePrice Includes Vat' => null,
                    'PurchaseCondition' => ($vehicle->is_new) ? 'new' : 'used',
                    'Website' => Site::vehicleUrl($vehicle),
                    'Origin' => null,
                    'V5' => 'true',
                    'ExDemo' => null,
                    'Franchise Approved' => null,
                    'PictureUrls' => $vehicle->specs
                        ->pluck('description')
                        ->implode(','),
                    'Service History' => 'Full',
                    'Four Wheel Drive' => null,
                    'Attention Grabber' => null,
                    'Options' => '',
                    'Description' => $vehicle->description,
                    'CapId'  => $vehicle->cap_id,
                    'GlassID' => null,
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'Feed_Id', //mandatory
            'Vehicle_ID', //mandatory
            'VRM', //mandatory
            'VRMYear',
            'VIN',
            'Date First Registered',
            'MOT Expiry Date',
            'Make', //mandatory
            'Model', //mandatory
            'Variant',
            'VehicleType', //mandatory
            'Mileage', //mandatory
            'BodyType', //mandatory
            'Doors',
            'Colour',
            'FuelType',
            'Engine Size',
            'Power Output',
            'CO2 (g/km)',
            'Transmission',
            'Previous Owners',
            'Price', //mandatory
            'Price Includes Vat', //mandatory
            'TradePrice',
            'TradePrice Includes Vat',  //mandatory IF TradePrice
            'PurchaseCondition',  //mandatory ( = new or used)
            'Website',
            'Origin',
            'V5',
            'ExDemo',
            'Franchise Approved',
            'PictureUrls', //mandatory
            'Service History',
            'Four Wheel Drive',
            'Attention Grabber',
            'Options', //mandatory
            'Description', //mandatory
            'CapId',
            'GlassID',
        ];
    }
}
