<?php

namespace App\Exports;

use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class PhyronExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
                'drivetrain',
                'specs',
                'equipment',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::PHYRON),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::PHYRON)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'dealerid' => $vehicle->dealership?->data['phyron-dealer-id'],
                    'vehicleid' => $vehicle->uuid,
                    'regnumber' => str_replace(' ', '', $vehicle->registration_number),
                    'colour' => $vehicle->colour,
                    'fueltype' => $vehicle->fuelType?->name,
                    'year' => $vehicle->manufacture_year,
                    'mileage' => $vehicle->odometer_mi,
                    'bodytype' => $vehicle->bodyStyle?->name,
                    'doors' => $vehicle->door_count,
                    'make' => $vehicle->make?->name,
                    'model' => $vehicle->model?->name,
                    'variant' => $vehicle->derivative,
                    'enginesize' => $vehicle->engine_size_cc,
                    'price' => $vehicle->price,
                    'transmissiontype' => $vehicle->transmission?->name,
                    'fourwheeldrive' => $vehicle->drivetrain?->name == 'Four Wheel Drive' ? 'Y' : 'N',
                    'images' => $vehicle->mediaUses
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large')),
                    'features' => $vehicle->specs->count()
                        ? $vehicle->specs->pluck('description')
                        : $vehicle->equipment->pluck('description'),
                    'description' => $vehicle->description,
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'dealer_ID',
            'Vehicle_ID',
            'FullRegistration',
            'Colour',
            'FuelType',
            'Year',
            'Mileage',
            'Bodytype',
            'Doors',
            'Make',
            'Model',
            'Variant',
            'EngineSize',
            'Price',
            'Transmission',
            'FourWheelDrive',
            'Images',
            'Options',
            'Description',
        ];
    }
}
