<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\MercuryDataModels\VehicleAttribute;
use Tests\TestCase;
use Tests\UserForTenant;

class VehicleAttributeControllerTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    protected $tenancy = true;
    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        VehicleAttribute::factory(4)->create(['model' => 'vehicle']);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.attributes.index', 'vehicle'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertCount(4, $response->json('data'));
    }

    public function testStore()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.attributes.store', 'vehicle'), [
                'name' => 'foo baz bar',
                'type' => 'number',
            ]);

        $response->assertStatus(201);

        $this->assertEquals('foo baz bar', $response->json('attribute.name'));
        $this->assertTrue(VehicleAttribute::query()->where('name', 'foo baz bar')->exists());
        $this->assertTrue(VehicleAttribute::query()->where('id', $response->json('attribute.id'))->exists());
    }

    public function testShow()
    {
        $attribute = VehicleAttribute::factory()->create([
            'model' => 'vehicle',
            'name' => 'foo baz bar',
            'type' => 'number',
        ]);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.attributes.show', ['vehicle', $attribute]));

        $response->assertStatus(200);

        $this->assertEquals('foo baz bar', $response->json('attribute.name'));
        $this->assertEquals('number', $response->json('attribute.type'));
    }

    public function testUpdate()
    {
        $attribute = VehicleAttribute::factory()->create(['model' => 'vehicle']);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.attributes.update', ['vehicle', $attribute]), [
                'name' => 'lorem ipsum',
                'type' => 'number',
                'count' => 3,
            ]);

        $response->assertStatus(200);

        $attribute->refresh();
        $this->assertEquals('lorem ipsum', $attribute->name);
        $this->assertEquals('number', $attribute->type);
        $this->assertEquals(3, $attribute->count);
    }


    public function testDestroy()
    {
        $attribute = VehicleAttribute::factory()->create(['model' => 'vehicle']);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->deleteJson(route('tenant.attributes.destroy', ['vehicle', $attribute]));

        $response->assertStatus(200);
        $this->assertFalse(VehicleAttribute::query()->where('id', $attribute->id)->exists());
    }
}
