<?php

namespace Tests\Unit;

use App\Facades\Settings;
use App\Modules\Sales\Services\SilverBullet;
use Database\Seeders\Tenant\SilverBulletSettingSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Storage;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Tests\TenantTestCase;
use Tests\TestCase;

class SilverBulletEcommerceTest extends TenantTestCase
{
    public function testGetEcommerceData()
    {
        $this->seed(SilverBulletSettingSeeder::class);
        Settings::update('sales-silver-bullet-domain', 'http://foo.baz/');
        Settings::update('sales-silver-bullet-site-name', 'bar');
        Settings::update('sales-silver-bullet-campaign-name', 'lorem');
        Settings::update('sales-silver-bullet-term', 50);
        Settings::update('sales-silver-bullet-deposit', 10);
        Settings::update('sales-silver-bullet-annual-mileage', 10000);
        $dealership = Dealership::factory()->create(['location_stock' => 123]);
        $vehicle = Vehicle::factory()->create(['dealership_id' => $dealership->id ]);

        $response = App::make(SilverBullet::class)->ecommerceData($vehicle, 'FPA');
        $this->assertEquals(
            'http://foo.baz/?utm_source=bar&utm_medium=FPA&utm_campaign=lorem',
            $response['storefront_base_url']
        );
        $this->assertEquals($vehicle->id, $response['vehicle_id']);
        $this->assertEquals(123, $response['dealer_id']);
        $this->assertEquals(50, $response['term']);
        $this->assertEquals(10, $response['deposit']);
        $this->assertEquals(10000, $response['annual_mileage']);
    }

    public function testGetDealUrl()
    {
        $this->seed(SilverBulletSettingSeeder::class);
        Settings::update('sales-silver-bullet-domain', 'http://foo.baz/');
        Settings::update('sales-silver-bullet-site-name', 'bar');
        Settings::update('sales-silver-bullet-campaign-name', 'lorem');

        $response = App::make(SilverBullet::class)->dealUrl('FPA');
        $this->assertEquals('http://foo.baz/?utm_source=bar&utm_medium=FPA&utm_campaign=lorem', $response);
    }

    public function testExport()
    {
        $this->seed(SilverBulletSettingSeeder::class);
        Settings::update('sales-silver-bullet-export-filename', 'lorem.csv');
        Storage::fake('file-storage');
        Storage::fake('silver-bullet');

        $vehicles = Vehicle::factory(5)->create([
            'dealership_id' => Dealership::factory()->create(['location_stock' => 'foo'])->id,
            'make_id' => VehicleMake::factory()->create()->id,
            'model_id' => VehicleModel::factory()->create()->id,
            'body_style_id' => BodyStyleType::factory()->create()->id,
            'transmission_id' => TransmissionType::factory()->create()->id,
            'fuel_type_id' => FuelType::factory()->create()->id,
        ]);

        $this->assertTrue(App::make(SilverBullet::class)->export($vehicles));
    }
}
