<?php

namespace App\Http\Resources;

use App\Events\LoadDealershipDataFields;
use App\ImportMaps\VehicleImportMap;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ImportConditionResource extends JsonResource
{
    public static $wrap = '';
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $this->resource->load('rules');
        $resource = $this->resource->toArray();
        $resource['provider_name'] = $this->resource->getProviderName();
        $resource['import_map_name'] = $this->resource->importMap?->name ?? __('labels.all');
        return $resource;
    }

    private function getFieldsForType(): array
    {
        return match ($this->resource->type) {
            'vehicle' => (new VehicleImportMap())->columns(),
            default => [],
        };
    }

    private function getDealershipFields()
    {
        return collect(event(new LoadDealershipDataFields()))
            ->filter()
            ->mapWithKeys(fn($field) => $field)
            ->map(function ($field, $key) {
                $field['id'] = $key;
                return $field;
            })
            ->prepend(['id' => 'location_stock', 'label' => 'Stock location'])
            ->values();
    }

    private function mapFields(): Collection
    {
        return $this->resource->fields->keyBy('on_file')
            ->map(function ($field) {
                $field['required_to_import'] = $field['data']['required_to_import'] ?? false;
                $field['save_on_update'] = $field['data']['save_on_update'] ?? true;
                if (!isset($field['data'])) {
                    $field['data'] = [];
                }
                return $field;
            });
    }

    private function mapExtraFields(array $extraFields = []): array
    {
        return collect($extraFields)
            ->map(function ($extraField) {
                $extraField['save_on_update'] = $extraField['save_on_update'] ?? true;
                return $extraField;
            })
            ->toArray();
    }

    private function loadRemoteFiles(): Collection
    {
        if (empty($this->resource->data['automate'])) {
            return collect();
        }

        try {
            $disk = Storage::build([
                'driver' => $this->resource['data']['protocol'],
                'root' => $this->resource['data']['file_root'] ?? null,
                'host' => $this->resource['data']['host'],
                'username' => $this->resource['data']['username'],
                'password' => $this->resource['data']['password'],
                'throw' => false,
                'timeout' => 2,
            ]);

            return collect($disk->files())
                ->map(fn($file) => [
                    'name' => $file,
                    'size' => $this->fileSize($disk, $file),
                ]);
        } catch (\Exception $exception) {
            return collect();
        }
    }

    private function fileSize($disk, string $file): string
    {
        $kb = $disk->size($file) / 1024;
        if ($kb < 1024) {
            return round($kb) . 'KiB';
        }

        return round($kb / 1024, 1) . 'MiB';
    }
}
