<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\VehicleValuation\Facades\Valuation;

class ValuationResource extends JsonResource
{
    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->load([
            'user',
            'enquiry',
            'adjustment.rules',
        ]);

        $this->resource->manufacture_year = Carbon::parse($this->resource->date_of_registration)->format('Y');
        $this->resource->preview_image = $this->previewImage();
        $this->resource->has_future_valuations = Valuation::hasEnabledDriver() && Valuation::hasFutureValuations();
        $this->resource->future_graph = $this->getFutureValueGraph();
        return parent::toArray($request);
    }

    public function previewImage(): string
    {
        return 'https://cdn.imagin.studio/getImage?' . http_build_query([
            'customer' => Settings::get('imagin-placeholders-client-key') ?: config('services.imagin-studio.key'),
            'make' => $this->resource->make,
            'modelFamily' => $this->resource->model,
            'paintdescription' => $this->resource->colour,
            'modelYear' => $this->resource->manufacture_year,
            'aspectRatio' => '16:9',
            'zoomLevel' => 50,
        ]);
    }

    private function getFutureValueGraph(): array|null
    {
        if (empty($this->resource->data['future_value'])) {
            return null;
        }

        return [
            'labels' => collect($this->resource->data['future_value'] ?? [])
                ->map(fn($item, $key)  => $key)
                ->values(),
            'datasets' => [
                [
                    'label' => 'Retail value',
                    'borderColor' => '#3498db',
                    'backgroundColor' => '#3498db',
                    'cubicInterpolationMode' => 'monotone',
                    'data' => collect($this->resource->data['future_value'] ?? [])
                        ->map(fn($item) => $item['retail'])
                        ->values(),
                ],
                [
                    'label' => 'Part Exchange value',
                    'borderColor' => '#f0ae49',
                    'backgroundColor' => '#f0ae49',
                    'cubicInterpolationMode' => 'monotone',
                    'data' => collect($this->resource->data['future_value'] ?? [])
                        ->map(fn($item) => $item['partExchange'])
                        ->values(),
                ],
                [
                    'label' => 'Trade value',
                    'borderColor' => '#7b7d83',
                    'backgroundColor' => '#7b7d83',
                    'cubicInterpolationMode' => 'monotone',
                    'data' => collect($this->resource->data['future_value'] ?? [])
                        ->map(fn($item) => $item['trade'])
                        ->values(),
                ],
                [
                    'label' => 'Private',
                    'borderColor' => '#c32aa3',
                    'backgroundColor' => '#c32aa3',
                    'cubicInterpolationMode' => 'monotone',
                    'data' => collect($this->resource->data['future_value'] ?? [])
                        ->map(fn($item) => $item['private'])
                        ->values(),
                ],
            ],
        ];
    }
}
