<?php

namespace App\Crm;

use App\Contracts\AddsDealershipFields;
use App\Contracts\ForwardsReservationToCrm;
use App\Facades\Settings;
use App\Services\WebLeadsApi;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\VehicleReservations\Reservation;

class SendToWebLeads implements EnquiryAction, ForwardsReservationToCrm, AddsDealershipFields
{
    use CheckForExpiredAction;

    public function __construct(private readonly WebLeadsApi $api)
    {
        //
    }

    public function enabled(): bool
    {
        return Settings::get('crm-webleads-enabled') === true;
    }

    /**
     * Data fields to add to form
     *
     * @return array
     */
    public function formAdditionalDataFields(): array
    {
        return [];
    }

    /**
     * Data fields to add to questions
     *
     * @return array[]
     */
    public function formQuestionDataFields(): array
    {
        return [
            'webleads-field' => [
                'type' => 'select',
                'label' => 'Field mapped against WebLeads CRM',
                'choices' => [
                    'title' => 'Title',
                    'firstname' => 'First Name',
                    'surname' => 'Last Name',
                    'email' => 'Email Address',
                    'phone' => 'Phone Number',
                    'mobile' => 'Mobile Number',
                    'notes' => 'Notes',
                ],
            ]
        ];
    }

    /**
     * Data fields to add to dealership
     *
     * @return array
     */
    public function dealershipAdditionalDataFields(): array
    {
        return [
            'webleads-dealer-id' => [
                'type' => 'text',
                'label' => 'WebLeads Dealer ID',
            ],
        ];
    }

    /**
     * Handle the action
     *
     * @param EnquiryActionModel $action
     * @param EnquiryModel $enquiry
     * @return bool|null
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        if ($this->expired($action)) {
            return null;
        }

        return $this->api->sendLead($enquiry, $action);
    }

    public function sendReservation(Reservation $reservation): bool
    {
        if ($this->api->sendReservation($reservation)) {
            $reservation->update(['crm_reference' => $this->api->getResponseAttribute('lead_id')]);
            return true;
        }
        return false;
    }

    /**
     * Message when successfully processed
     *
     * @return string
     */
    public function successDetails(): string
    {
        return $this->api->getResponseMessage();
    }

    /**
     * Message when failure detected
     *
     * @return string
     */
    public function failureReason(): string
    {
        return $this->api->getResponseAttribute('error') ?? $this->api->getResponseMessage() ?? 'Unknown error';
    }
}
