<?php

namespace App\Services;

use App\Crm\Config\WebLeadsConfig;
use App\Crm\EnquiryDataToProviderData;
use Carbon\Carbon;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\VehicleValuation\Models\VehicleValuation;

class WebLeadsApi
{
    use EnquiryDataToProviderData;

    private Response $response;

    public function __construct(protected readonly WebLeadsConfig $config)
    {
        //
    }

    public function getResponseAttribute(string $attribute): mixed
    {
        return $this->response->json($attribute);
    }

    public function getResponseMessage(): ?string
    {
        return $this->response->json('message');
    }

    public function sendLead(EnquiryModel $enquiry, EnquiryActionModel $action): bool
    {
        return $this->submitLead($this->mapEnquiry($enquiry));
    }

    protected function mapEnquiry(EnquiryModel $enquiry): array
    {
        $params = $this->convertEnquiryToParams($enquiry, 'webleads-field');

        foreach ($enquiry->objects as $object) {
            if ($object->object_type === 'vehicle' && $object->object) {
                $params['vehicle'] = $this->mapVehicle($object->object);
            }

            if ($object->object_type === 'valuation' && $object->object) {
                $params['valuation'] = $this->mapValuation($object->object);
            }
        }

        // $params['website'] = Site::url();
        // $params['source_url'] = Site::url($enquiry->data['meta']['source_url'] ?? '');

        $data = [
            'enquiry' => [
                'apikey' => $this->config->apiKey(),
                'lead' => [
                    'origin' => 'todo-origin',
                    'source' => 'todo-source',
                    'branch' => $params['vehicle']['branch_id'] ?? $this->config->fallbackDealerId(),
                    'idnumber' => $enquiry->id,
                ],
                'contact' => [
                    'firstname' => $params['firstname'] ?? '',
                    'surname' => $params['surname'] ?? '',
                    'email' => $params['email'] ?? '',
                    'phone' => $params['phone'] ?? '',
                    'mobile' => $params['mobile'] ?? '',
                    'source' => 'todo-source'
                ],
                'seeks' => [
                    'newused' => $params['vehicle']['new_used'] ?? '',
                    'make' => $params['vehicle']['make'] ?? '',
                    'model' => $params['vehicle']['model'] ?? '',
                    'stocknumber' => $params['vehicle']['uuid'] ?? '',
                    'regnumber' => $params['vehicle']['vrm'] ?? '',
                    'notes' => $params['notes'] ?? '',
                ],
                'tradein' => [
                    'registration' => $params['valuation']['registration'] ?? '',
                    'make' => $params['valuation']['make'] ?? '',
                    'model' => $params['valuation']['model'] ?? '',
                    'mileage' => $params['valuation']['mileage'] ?? '',
                    'year' => $params['valuation']['year'] ?? '',
                    'notes' => $params['notes'] ?? '',
                ],
            ],
            'mkagree' => [
                // TODO: check assumptions e.g. about marketing opt in
                // TODO: check if these should be booleans
                'updatemkagree' => 'agree-to-marketing',
                'dealersms' => null,
                'dealeremail' => null,
                'dealerletter' => null,
                'dealerphone' => null,
                'partssms' => 'agree-to-marketing-sms',
                'partsemail' => 'agree-to-marketing-email',
                'partsletter' => null,
                'partsphone' => null,
                'salessms' => 'agree-to-marketing-sms',
                'salesemail' => 'agree-to-marketing-email',
                'salesletter' => null,
                'salesphone' => null,
                'servicesms' => 'agree-to-marketing-sms',
                'serviceemail' => 'agree-to-marketing-email',
                'serviceletter' => null,
                'servicephone' => null
            ],
        ];

        return $data;
    }

    protected function mapVehicle(Vehicle $vehicle): array
    {
        return [
            'make' => $vehicle->make?->name,
            'model' => $vehicle->model?->name,
            'vrm' => $vehicle->vrm_condensed,
            'uuid' => $vehicle->uuid,
            // TODO: get accepted new / used value
            'new_used' => 'todo-newused',
            'branch_id' => $vehicle->dealership?->data['webleads-dealer-id'] ?? null,
        ];
    }

    protected function mapValuation(VehicleValuation $valuation): array
    {
        return [
            'registration' => $valuation->registration ?? '',
            'make' => $valuation->make ?? '',
            'model' => $valuation->model ?? '',
            'mileage' => $valuation->mileage ?? '',
            'year' => $valuation->date_of_registration
                ? Carbon::parse($valuation->date_of_registration)->format('Y')
                : '',
        ];
    }

    protected function submitLead(array $leadData): bool
    {
        $endpoint = $this->endpoint();
        $this->response = Http::post($endpoint, $leadData);

        ApiNotification::query()->create([
            'provider' => 'webleads',
            'processed' => $this->response->successful(),
            'headers' => null,
            'data' => [
                'request' => $leadData,
                'result' => $this->response->body(),
                'endpoint' => $endpoint,
            ],
            'data_model' => WebLeadsApi::class,
        ]);

        return $this->response->successful();
    }

    private function endpoint(): string
    {
        return rtrim($this->config->apiDomain(), '/')
            . '/tramo/leads-v1/generic.json?SoapAction=InboundEnquiry&ContentType=application/json';
    }
}
