<?php

namespace Tests\Tenant;

use App\Crm\Config\WebLeadsConfig;
use App\Facades\Settings;
use App\Master\Models\VehicleMake;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\EnquiryAction;
use Mtc\Crm\Models\FormQuestion;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleValuation;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class WebLeadsCrmTest extends TenantTestCase
{
    use UserForTenant;

    public function testConfig()
    {
        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'WebLeads',
            'name' => 'Enabled',
            'config_key' => 'crm-webleads-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'WebLeads',
            'name' => 'API Key',
            'config_key' => 'crm-webleads-api-key',
            'type' => 'string',
            'value' => 'foo_api_key',
        ]);

        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'WebLeads',
            'name' => 'API Domain',
            'config_key' => 'crm-webleads-api-domain',
            'type' => 'string',
            'value' => 'https://foo.com',
        ]);

        /** @var WebLeadsConfig $config */
        $config = App::make(WebLeadsConfig::class);
        $this->assertTrue($config->enabled());
        $this->assertEquals('https://foo.com', $config->apiDomain());
        $this->assertEquals('foo_api_key', $config->apiKey());
    }

    public function testSendLead()
    {
        $this->addSettingsAndHttpFake();

        $q1 = FormQuestion::factory()->create(['data' => ["webleads-field" => "firstname"]]);
        $q2 = FormQuestion::factory()->create(['data' => ["webleads-field" => "surname"]]);
        $q3 = FormQuestion::factory()->create(['data' => ["webleads-field" => "phone"]]);
        $q4 = FormQuestion::factory()->create(['data' => ["webleads-field" => "email"]]);
        $q5 = FormQuestion::factory()->create(['data' => ["webleads-field" => "notes"]]);

        $enquiry = Enquiry::factory()->create([
            'email' => 'foo@bar.com',
            'details' => [
                $q1->id => ['answer' => 'Peter'],
                $q2->id => ['answer' => 'Parker'],
                $q3->id => ['answer' => '00000111222'],
                $q4->id => ['answer' => 'foo@bar.com'],
                $q5->id => ['answer' => 'my_message'],
            ],
            'source' => 'test.com',
        ]);

        $action = EnquiryAction::factory()->create([
            'action_name' => 'send-to-webleads',
            'data' => [],
            'processed_at' => null,
            'failed_at' => null
        ]);

        $enquiry_action = Enquiries::makeAction('send-to-webleads');
        $enquiry_action->handle($action, $enquiry);

        Http::assertSent(function (Request $request) use ($enquiry) {
            return Str::contains($request->url(), 'https://foo.com/tramo/leads-v1/generic.json')
                && $request['enquiry']['apikey'] === 'foo_api_key'
                && $request['enquiry']['contact']['firstname'] === 'Peter'
                && $request['enquiry']['contact']['surname'] === 'Parker'
                && $request['enquiry']['contact']['email'] === 'foo@bar.com'
                && $request['enquiry']['contact']['phone'] === '00000111222'
                && $request['enquiry']['seeks']['notes'] === 'my_message'
                && $request['enquiry']['lead']['idnumber'] === $enquiry->id
                && $request['enquiry']['lead']['branch'] === 'fallback-dealer-123';
        });

        $this->assertCount(1, ApiNotification::query()->where('provider', 'webleads')->get());
    }

    public function testSendLeadForVehicle()
    {
        $this->addSettingsAndHttpFake();

        $q1 = FormQuestion::factory()->create(['data' => ["webleads-field" => "firstname"]]);
        $q2 = FormQuestion::factory()->create(['data' => ["webleads-field" => "surname"]]);
        $q3 = FormQuestion::factory()->create(['data' => ["webleads-field" => "phone"]]);
        $q4 = FormQuestion::factory()->create(['data' => ["webleads-field" => "email"]]);
        $q5 = FormQuestion::factory()->create(['data' => ["webleads-field" => "notes"]]);

        $make = VehicleMake::factory()->create(['name' => 'Audi']);
        $model = VehicleModel::factory()->create(['name' => 'A6']);
        $dealership = Dealership::factory()->create([
            'data' => ['webleads-dealer-id' => 'dealer-456']
        ]);
        $vehicle = Vehicle::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership->id,
            'type' => 'car',
            'is_new' => 0,
            'price' => 12345,
            'registration_number' => 'FO25 FAZ',
            'vrm_condensed' => 'FO25FAZ',
            'uuid' => 'test-uuid-123',
        ]);

        $enquiry = Enquiry::factory()->create([
            'email' => 'foo@bar.com',
            'details' => [
                $q1->id => ['answer' => 'Peter'],
                $q2->id => ['answer' => 'Parker'],
                $q3->id => ['answer' => '00000111222'],
                $q4->id => ['answer' => 'foo@bar.com'],
                $q5->id => ['answer' => 'my_message'],
            ],
            'source' => 'test.com',
        ]);
        $enquiry->objects()->create([
            'object_id' => $vehicle->id,
            'object_type' => $vehicle->getMorphClass(),
        ]);

        $action = EnquiryAction::factory()->create([
            'action_name' => 'send-to-webleads',
            'data' => [],
            'processed_at' => null,
            'failed_at' => null
        ]);

        $enquiry_action = Enquiries::makeAction('send-to-webleads');
        $enquiry_action->handle($action, $enquiry);

        Http::assertSent(function (Request $request) {
            return Str::contains($request->url(), 'https://foo.com/tramo/leads-v1/generic.json')
                && $request['enquiry']['apikey'] === 'foo_api_key'
                && $request['enquiry']['contact']['firstname'] === 'Peter'
                && $request['enquiry']['contact']['surname'] === 'Parker'
                && $request['enquiry']['contact']['email'] === 'foo@bar.com'
                && $request['enquiry']['contact']['phone'] === '00000111222'
                && $request['enquiry']['seeks']['make'] === 'Audi'
                && $request['enquiry']['seeks']['model'] === 'A6'
                && $request['enquiry']['seeks']['regnumber'] === 'FO25FAZ'
                && $request['enquiry']['seeks']['stocknumber'] === 'test-uuid-123'
                && $request['enquiry']['lead']['branch'] === 'dealer-456';
        });

        $this->assertCount(1, ApiNotification::query()->where('provider', 'webleads')->get());
    }

    public function testSendLeadForVehicleWithFallbackDealer()
    {
        $this->addSettingsAndHttpFake();

        $q1 = FormQuestion::factory()->create(['data' => ["webleads-field" => "firstname"]]);
        $q2 = FormQuestion::factory()->create(['data' => ["webleads-field" => "email"]]);

        $make = VehicleMake::factory()->create(['name' => 'Audi']);
        $model = VehicleModel::factory()->create(['name' => 'A6']);
        // Dealership without webleads-dealer-id
        $dealership = Dealership::factory()->create([
            'data' => []
        ]);
        $vehicle = Vehicle::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership->id,
            'type' => 'car',
            'is_new' => 0,
            'price' => 12345,
            'vrm_condensed' => 'FO25FAZ',
        ]);

        $enquiry = Enquiry::factory()->create([
            'email' => 'foo@bar.com',
            'details' => [
                $q1->id => ['answer' => 'Peter'],
                $q2->id => ['answer' => 'foo@bar.com'],
            ],
            'source' => 'test.com',
        ]);
        $enquiry->objects()->create([
            'object_id' => $vehicle->id,
            'object_type' => $vehicle->getMorphClass(),
        ]);

        $action = EnquiryAction::factory()->create([
            'action_name' => 'send-to-webleads',
            'data' => [],
            'processed_at' => null,
            'failed_at' => null
        ]);

        $enquiry_action = Enquiries::makeAction('send-to-webleads');
        $enquiry_action->handle($action, $enquiry);

        Http::assertSent(function (Request $request) {
            // When no dealer ID on dealership, fallback should be used
            return Str::contains($request->url(), 'https://foo.com/tramo/leads-v1/generic.json')
                && $request['enquiry']['lead']['branch'] === 'fallback-dealer-123';
        });

        $this->assertCount(1, ApiNotification::query()->where('provider', 'webleads')->get());
    }

    public function testSendLeadForValuation()
    {
        $this->addSettingsAndHttpFake();

        $q1 = FormQuestion::factory()->create(['data' => ["webleads-field" => "firstname"]]);
        $q2 = FormQuestion::factory()->create(['data' => ["webleads-field" => "surname"]]);
        $q3 = FormQuestion::factory()->create(['data' => ["webleads-field" => "phone"]]);
        $q4 = FormQuestion::factory()->create(['data' => ["webleads-field" => "email"]]);
        $q5 = FormQuestion::factory()->create(['data' => ["webleads-field" => "notes"]]);

        $valuation = VehicleValuation::query()->create([
            'provider' => 'foo',
            'registration' => 'FO25FAZ',
            'mileage' => 25000,
            'make' => 'Ford',
            'model' => 'Focus',
            'average_price' => 99448,
            'cosmetic_condition' => 'good',
            'date_of_registration' => '2020-06-15',
        ]);

        $enquiry = Enquiry::factory()->create([
            'email' => 'foo@bar.com',
            'details' => [
                $q1->id => ['answer' => 'Peter'],
                $q2->id => ['answer' => 'Parker'],
                $q3->id => ['answer' => '00000111222'],
                $q4->id => ['answer' => 'foo@bar.com'],
                $q5->id => ['answer' => 'my_message'],
            ],
            'source' => 'test.com',
        ]);
        $enquiry->objects()->create([
            'object_id' => $valuation->id,
            'object_type' => $valuation->getMorphClass(),
        ]);

        $action = EnquiryAction::factory()->create([
            'action_name' => 'send-to-webleads',
            'data' => [],
            'processed_at' => null,
            'failed_at' => null
        ]);

        $enquiry_action = Enquiries::makeAction('send-to-webleads');
        $enquiry_action->handle($action, $enquiry);

        Http::assertSent(function (Request $request) {
            return Str::contains($request->url(), 'https://foo.com/tramo/leads-v1/generic.json')
                && $request['enquiry']['tradein']['registration'] === 'FO25FAZ'
                && $request['enquiry']['tradein']['make'] === 'Ford'
                && $request['enquiry']['tradein']['model'] === 'Focus'
                && $request['enquiry']['tradein']['mileage'] == 25000
                && $request['enquiry']['tradein']['year'] === '2020';
        });

        $this->assertCount(1, ApiNotification::query()->where('provider', 'webleads')->get());
    }

    public function testSendLeadWithVehicleAndValuation()
    {
        $this->addSettingsAndHttpFake();

        $q1 = FormQuestion::factory()->create(['data' => ["webleads-field" => "firstname"]]);
        $q2 = FormQuestion::factory()->create(['data' => ["webleads-field" => "email"]]);

        $make = VehicleMake::factory()->create(['name' => 'Audi']);
        $model = VehicleModel::factory()->create(['name' => 'A6']);
        $dealership = Dealership::factory()->create([
            'data' => ['webleads-dealer-id' => 'dealer-456']
        ]);
        $vehicle = Vehicle::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership->id,
            'type' => 'car',
            'is_new' => 0,
            'registration_number' => 'FO25 FAZ',
            'uuid' => 'test-uuid-123',
        ]);

        $valuation = VehicleValuation::query()->create([
            'provider' => 'foo',
            'registration' => 'XX25YYY',
            'mileage' => 50000,
            'make' => 'Ford',
            'model' => 'Focus',
            'average_price' => 8000,
            'date_of_registration' => '2019-03-10',
        ]);

        $enquiry = Enquiry::factory()->create([
            'email' => 'foo@bar.com',
            'details' => [
                $q1->id => ['answer' => 'Peter'],
                $q2->id => ['answer' => 'foo@bar.com'],
            ],
            'source' => 'test.com',
        ]);
        $enquiry->objects()->create([
            'object_id' => $vehicle->id,
            'object_type' => $vehicle->getMorphClass(),
        ]);
        $enquiry->objects()->create([
            'object_id' => $valuation->id,
            'object_type' => $valuation->getMorphClass(),
        ]);

        $action = EnquiryAction::factory()->create([
            'action_name' => 'send-to-webleads',
            'data' => [],
            'processed_at' => null,
            'failed_at' => null
        ]);

        $enquiry_action = Enquiries::makeAction('send-to-webleads');
        $enquiry_action->handle($action, $enquiry);

        Http::assertSent(function (Request $request) {
            return Str::contains($request->url(), 'https://foo.com/tramo/leads-v1/generic.json')
                // Vehicle data in seeks
                && $request['enquiry']['seeks']['make'] === 'Audi'
                && $request['enquiry']['seeks']['model'] === 'A6'
                && $request['enquiry']['seeks']['regnumber'] === 'FO25FAZ'
                // Valuation data in tradein
                && $request['enquiry']['tradein']['registration'] === 'XX25YYY'
                && $request['enquiry']['tradein']['make'] === 'Ford'
                && $request['enquiry']['tradein']['model'] === 'Focus'
                && $request['enquiry']['tradein']['mileage'] == 50000
                && $request['enquiry']['tradein']['year'] === '2019';
        });

        $this->assertCount(1, ApiNotification::query()->where('provider', 'webleads')->get());
    }

    private function addSettingsAndHttpFake()
    {
        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'WebLeads',
            'name' => 'Enabled',
            'config_key' => 'crm-webleads-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'WebLeads',
            'name' => 'API Key',
            'config_key' => 'crm-webleads-api-key',
            'type' => 'string',
            'value' => 'foo_api_key',
        ]);

        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'WebLeads',
            'name' => 'API Domain',
            'config_key' => 'crm-webleads-api-domain',
            'type' => 'string',
            'value' => 'https://foo.com',
        ]);

        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'WebLeads',
            'name' => 'Fallback Dealer ID',
            'config_key' => 'crm-webleads-fallback-dealer-id',
            'type' => 'string',
            'value' => 'fallback-dealer-123',
        ]);

        Http::fake([
            '*' => Http::response([
                'message' => 'Lead created successfully',
                'lead_id' => 'WL123456',
            ], 200),
        ]);
    }
}
