<?php

namespace App\Imports\Traits;

use App\Facades\Settings;
use App\Jobs\ImportImagesFromUrlList;
use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;
use App\Models\ImportMap;
use App\Models\ImportMapField;
use App\TaxonomyMap;
use App\Traits\ImportChecksConditions;
use App\Traits\MapsTaxonomies;
use App\Traits\StockSyncTraits;
use App\VehicleType;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Mtc\ContentManager\Facades\Media;
use Mtc\ContentManager\Facades\Media as MediaFacade;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Country;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\DrivetrainType;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\TransmissionType;
use Symfony\Component\Finder\SplFileInfo;

/**
 * Base trait for import flows containing shared logic for both
 * Vehicle and VehicleOffer imports via ImportMap.
 */
trait BaseImportFlow
{
    use ImportChecksConditions;
    use MapsTaxonomies;
    use StockSyncTraits;

    protected string $subFieldDelimiter = '|';
    protected static array $allowedFieldsToUpdate = [];
    protected bool $stock_sync = false;
    protected ?string $archive_folder = null;
    protected ?ImportMap $import_map = null;
    protected string $dealershipMatchValue = 'location_stock';

    // Relationship caches
    protected Collection|null $dealerships = null;
    protected Collection|null $makes = null;
    protected Collection|null $models = null;
    protected Collection|null $fuel_types = null;
    protected Collection|null $body_styles = null;
    protected Collection|null $drivetrains = null;
    protected Collection|null $transmissions = null;

    public function setAdditionalTasks(array $tasks): self
    {
        $this->additional_tasks = $tasks;
        return $this;
    }

    public function setImportMap(ImportMap $importMap, bool $stockSync = false): self
    {
        $this->import_map = $importMap;
        $this->stock_sync = $stockSync;
        if (!empty($this->import_map->data['sub_field_delimiter'])) {
            $this->subFieldDelimiter = $this->import_map->data['sub_field_delimiter'];
        }
        return $this;
    }

    public function setArchiveFolder(string $archive_folder): self
    {
        $this->archive_folder = $archive_folder;
        return $this;
    }

    protected function getArchiveFolder(): string
    {
        if (empty($this->archive_folder)) {
            $this->archive_folder = uniqid('zip_extract_', true);
        }

        return $this->archive_folder;
    }

    protected function getProviderName(): string
    {
        if (empty($this->import_map)) {
            return $this->importType;
        }

        return 'import-map-' . $this->import_map->id;
    }

    /**
     * Load base relationships common to all import types.
     */
    protected function loadBaseRelationships(): void
    {
        $this->dealerships = Dealership::all();
        $this->makes = VehicleMake::all()->pluck('id', 'name');
        $this->models = VehicleModel::all()->pluck('id', 'name');
        $this->fuel_types = FuelType::all()->pluck('id', 'name');
        $this->body_styles = BodyStyleType::all()->pluck('id', 'name');
        $this->drivetrains = DrivetrainType::all()->pluck('id', 'name');
        $this->transmissions = TransmissionType::all()->pluck('id', 'name');
    }

    /**
     * Map source row data based on ImportMap field configuration.
     */
    protected function mapBasedOnImportMap(array $row): array
    {
        if (!isset($this->import_map)) {
            $this->import_map = ImportMap::query()->with('fields')->find($this->importType);
            if (!$this->import_map) {
                throw new Exception('Unrecognized import type: ' . $this->importType);
            }
        }

        if (!empty($this->import_map->data['dealership_field'])) {
            $this->dealershipMatchValue = $this->import_map->data['dealership_field'];
        }

        $data = $this->import_map->fields
            ->keyBy(fn(ImportMapField $field) => $field->on_model)
            ->filter(fn(ImportMapField $field, $key) => !empty($key))
            ->map(fn(ImportMapField $field) => $this->ensureFormattingForField(
                $field->on_model,
                $row[$field->on_file] ?? $this->decodeDefaultValue($field->default_value, $row),
                $row,
                $field->data['cast_type'] ?? null
            ))
            ->reject(fn($value) => is_null($value))
            ->toArray();

        return $this->verifyRowRequiredFields($data);
    }

    /**
     * Verify that all required fields are present and valid.
     */
    protected function verifyRowRequiredFields(array $data): array
    {
        $required_fields = $this->import_map->fields
            ->filter(fn(ImportMapField $field) => !empty($field->data['required_to_import']))
            ->pluck('on_model');

        $has_missing_required_fields = $required_fields
            ->filter(fn(string $field) => empty($data[$field]) || $this->rowFailsValidation($field, $data))
            ->isNotEmpty();

        return $has_missing_required_fields ? [] : $data;
    }

    /**
     * Check if a field value fails validation rules.
     * Override in child traits to add model-specific validation.
     */
    protected function rowFailsValidation(string $field, array $data): bool
    {
        return false;
    }

    /**
     * Decode default value which may contain logic blocks.
     */
    protected function decodeDefaultValue($default, array $row)
    {
        // If not logic block, return literal value
        if (!(is_string($default) && Str::startsWith($default, '{{') && Str::endsWith($default, '}}'))) {
            return $default;
        }

        // Strip logic identifiers
        $lookup = str_replace(['{{', '}}'], '', $default);
        $matched_operator = collect(['>=', '<=', '!=', '>', '<', '='])
            ->filter(fn($operator) => stripos($lookup, $operator) !== false)
            ->first();

        // Not a supported operator, return column with specified name or null
        if (!$matched_operator) {
            return $row[$lookup] ?? null;
        }

        $conditions = explode($matched_operator, $lookup);

        return match ($matched_operator) {
            '>=' => ($row[$conditions[0]] ?? '') >= $conditions[1],
            '<=' => ($row[$conditions[0]] ?? '') <= $conditions[1],
            '!=' => ($row[$conditions[0]] ?? '') != $conditions[1],
            '>' => ($row[$conditions[0]] ?? '') > $conditions[1],
            '<' => ($row[$conditions[0]] ?? '') < $conditions[1],
            '=' => ($row[$conditions[0]] ?? '') == $conditions[1],
            default => $conditions[0],
        };
    }

    /**
     * Get dealership ID from stock location value.
     */
    protected function getDealershipId($locationId): ?string
    {
        return $this->dealerships
            ->filter(fn(Dealership $dealership) => $dealership[$this->dealershipMatchValue] == $locationId
                || ($dealership->data[$this->dealershipMatchValue] ?? null) == $locationId)
            ->first()
            ?->id;
    }

    // =========================================================================
    // Casting Methods
    // =========================================================================

    /**
     * Cast a value based on ImportMap field configuration.
     */
    protected function castValue(string $field, $value, array $row): mixed
    {
        if (empty($this->import_map)) {
            return $value;
        }

        $fieldConfig = $this->import_map->fields
            ->filter(fn($entry) => $entry->on_model === $field)
            ->first();

        $castType = $fieldConfig?->data['cast_type'] ?? null;

        return match ($castType) {
            'bool' => $this->castToBoolean($value),
            'number' => (float) $value,
            'datetime' => $this->castToDateTime($value),
            'year_from_date' => $this->castToYearFromDate($value),
            'vehicle_type' => $this->castToVehicleType($value, $fieldConfig?->data['vehicle_type_mappings'] ?? []),
            default => $value,
        };
    }

    protected function castToBoolean($value): bool
    {
        if (!is_string($value)) {
            return (bool) $value;
        }

        return match (strtolower($value)) {
            '1', 'yes', 'true', 'y', 'on', 'new' => true,
            default => false,
        };
    }

    protected function castToDateTime($value): Carbon|string|null
    {
        if (empty($value)) {
            return null;
        }

        try {
            $date = $this->parseDateValue($value);
            return $date?->gt(Carbon::create(1950)) ? $date : null;
        } catch (Exception $exception) {
            return $value;
        }
    }

    protected function castToYearFromDate($value): ?int
    {
        try {
            $date = $this->parseDateValue($value);
            return $date?->year > 1950 ? $date->year : null;
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
            return null;
        }
    }

    protected function parseDateValue($value): ?Carbon
    {
        if (empty($value)) {
            return null;
        }

        $dateFormat = $this->import_map->data['date_format'] ?? null;

        if ($dateFormat === 'excel') {
            return $this->parseExcelDate($value);
        }

        if (!empty($dateFormat)) {
            return Carbon::createFromFormat($dateFormat, $value);
        }

        return Carbon::parse($value);
    }

    protected function parseExcelDate($value): ?Carbon
    {
        if (!is_numeric($value)) {
            return null;
        }

        // Excel serial date: days since December 30, 1899
        $excelEpoch = Carbon::create(1899, 12, 30);
        return $excelEpoch->addDays((int) $value);
    }

    protected function castToVehicleType($value, array $mappings): string
    {
        $normalizedValue = strtolower(trim($value ?? ''));

        foreach ($mappings as $mapping) {
            if (strtolower(trim($mapping['from'] ?? '')) === $normalizedValue) {
                return $mapping['to'];
            }
        }

        return match ($normalizedValue) {
            'lcv' => VehicleType::LCV->value,
            'motorcycle' => VehicleType::MOTORCYCLE->value,
            default => VehicleType::CAR->value,
        };
    }

    // =========================================================================
    // Field Mapping Methods
    // =========================================================================

    /**
     * Map field names to model column names.
     * Override in child traits to add model-specific mappings.
     */
    protected function getMappedField(string $field): string
    {
        return match ($field) {
            'make' => 'make_id',
            'model' => 'model_id',
            'dealership' => 'dealership_id',
            'fuel_type' => 'fuel_type_id',
            'body_style' => 'body_style_id',
            'drivetrain' => 'drivetrain_id',
            'transmission' => 'transmission_id',
            default => $field,
        };
    }

    /**
     * Map field value with appropriate transformations.
     * Override in child traits to add model-specific value mappings.
     */
    protected function getMappedValue(string $field, $value, array $row)
    {
        return match ($field) {
            'make' => $value && !empty($this->makes[$value])
                ? $this->makes[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::MAKE, $value, $row),
            'model' => !empty($value) ? $this->getModelValue($value, $row) : null,
            'dealership' => $value && !empty($this->dealerships[$value])
                ? $this->dealerships[$value]
                : $this->getDealershipId($value),
            'fuel_type' => $value && !empty($this->fuel_types[$value])
                ? $this->fuel_types[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::FUEL_TYPE, $value, $row),
            'body_style' => $value && !empty($this->body_styles[$value])
                ? $this->body_styles[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::BODY_STYLE, $value, $row),
            'drivetrain' => $value && !empty($this->drivetrains[$value])
                ? $this->drivetrains[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::DRIVETRAIN, $value, $row),
            'transmission' => $value && !empty($this->transmissions[$value])
                ? $this->transmissions[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::TRANSMISSION, $value, $row),
            'featured' => $value ?? 0,
            default => $this->castValue($field, $value, $row),
        };
    }

    protected function getModelValue($value, array $row)
    {
        $cast_type = $this->import_map?->fields
            ->filter(fn($entry) => $entry->on_model === 'model')
            ->first()
            ?->data['cast_type'] ?? null;

        if ($cast_type === 'model_lookup') {
            $this->castToModelLookup($value, $row['make'] ?? null);
        }

        return $this->getMappedTaxonomy(
            TaxonomyMap::MODEL,
            $value,
            $row,
            $this->getMappedTaxonomy(TaxonomyMap::MAKE, $row['make'] ?? null, $row)
        );
    }

    /**
     * Ensure correct formatting for a field value.
     * Override in child traits to add model-specific formatting.
     */
    protected function ensureFormattingForField(
        string $on_model,
        $value,
        array $row,
        ?string $valueCastType = null
    ): mixed {
        return match ($on_model) {
            'make_id' => $this->getMappedTaxonomy(TaxonomyMap::MAKE, $value, $row),
            'model_id' => $this->getMappedTaxonomy(TaxonomyMap::MODEL, $value, $row),
            'transmission_id' => $this->getMappedTaxonomy(TaxonomyMap::TRANSMISSION, $value, $row),
            'body_style_id' => $this->getMappedTaxonomy(TaxonomyMap::BODY_STYLE, $value, $row),
            'drivetrain_id' => $this->getMappedTaxonomy(TaxonomyMap::DRIVETRAIN, $value, $row),
            'fuel_type_id' => $this->getMappedTaxonomy(TaxonomyMap::FUEL_TYPE, $value, $row),
            'dealership_id' => $this->getDealershipId($value),
            default => is_string($value) ? trim($value) : $value,
        };
    }

    // =========================================================================
    // Image Handling Methods
    // =========================================================================

    protected function imageAlreadyHostedOnServer($image): bool
    {
        try {
            return Media::assetExists($image);
        } catch (Exception $exception) {
            return false;
        }
    }

    protected function processImageFiles(array $images): void
    {
        collect(File::files(Storage::disk('local')->path($this->getArchiveFolder())))
            ->filter(fn(SplFileInfo $file) => in_array($file->getFilename(), $images))
            ->tap(fn(Collection $chunk) => $this->importImageBatch($chunk));
    }

    protected function importImageBatch(Collection $chunk): void
    {
        $file_names = $chunk->map(fn(SplFileInfo $file) => $file->getFilename());
        $existing = \Mtc\MercuryDataModels\Media::query()
            ->where('media.image_provider', $this->getProviderName())
            ->whereIn('media.source_filename', $file_names)
            ->pluck('media.source_filename');

        $chunk->reject(fn(SplFileInfo $file) => $existing->search($file->getFilename()) !== false)
            ->each(fn(SplFileInfo $file) => MediaFacade::importImageFromFile($file, $this->getProviderName()));
    }

    /**
     * Sync images for a model from URL list or file names.
     */
    protected function syncImagesForModel(Model $model, string $images): void
    {
        $provider = isset($this->import_map) && $this->stock_sync ? 'import-map-' . $this->import_map->id : null;
        $imageList = collect(explode($this->subFieldDelimiter, $images))->filter();

        // Only when doing system default, import maps rely on source_filename field
        if (is_null($provider)) {
            $imageList = $imageList->reject(fn(string $image) => $this->imageAlreadyHostedOnServer($image));
        }

        if ($imageList->isEmpty() && $provider !== null) {
            $this->removeProviderImages($model, $provider);
            return;
        }

        if ($imageList->isNotEmpty() && $this->imageChecksumMismatch($imageList->toArray(), $model)) {
            if (stripos($imageList->first(), 'http') === false) {
                $this->processImageFiles($imageList->toArray());
                $this->syncImagesFromFile($model, $imageList->toArray());
            } else {
                $this->dispatch(new ImportImagesFromUrlList(
                    $imageList,
                    $model,
                    false,
                    $provider,
                    true
                ));
            }
        }
    }

    protected function removeProviderImages(Model $model, string $provider): void
    {
        $model->mediaUses()->whereHas('media', fn($q) => $q->where('image_provider', $provider))->delete();

        if (method_exists($model, 'imageChecksumAttribute')) {
            $model[$model->imageChecksumAttribute()] = null;
            $model->save();
        }
    }

    protected function syncImagesFromFile(Model $model, array $images): void
    {
        $media = \Mtc\MercuryDataModels\Media::query()
            ->whereIn('source_filename', $images)
            ->where('image_provider', $this->getProviderName())
            ->pluck('id')
            ->toArray();

        MediaFacade::setUsesForModel($media, $model, ['primary' => true]);
    }

    /**
     * Sync features for a model.
     */
    protected function syncFeaturesForModel(Model $model, string $features): void
    {
        collect(explode($this->subFieldDelimiter, $features))
            ->filter()
            ->each(fn($feature) => $model->features()->updateOrCreate(['name' => $feature]));
    }
}
