<?php

namespace App\Imports\Traits;

use App\Events\NewVehicleImported;
use App\Events\VehicleUpdatedFromImport;
use App\Facades\Settings;
use App\TaxonomyMap;
use App\Traits\EnsuresVehicleAttribute;
use App\VehicleType;
use Illuminate\Support\Facades\Event;
use Mtc\MercuryDataModels\Country;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;

/**
 * Trait for importing Vehicles via ImportMap.
 * Uses BaseImportFlow for shared logic and adds vehicle-specific functionality.
 */
trait VehicleImportFlow
{
    use BaseImportFlow;
    use EnsuresVehicleAttribute;

    protected static array $vehicleFillable = [];
    protected array $vehicleAttributes = [];

    /**
     * Load relationships needed for vehicle imports.
     */
    protected function loadRelationships(): void
    {
        $this->loadBaseRelationships();
        $this->retrieveRequiredVehicleAttributes();
    }

    protected function retrieveRequiredVehicleAttributes(): void
    {
        $this->vehicleAttributes = VehicleAttribute::all()->mapWithKeys(fn($attribute) => [
            $attribute->slug => $attribute->toArray(),
        ])->toArray();
    }

    /**
     * Process a single row into a Vehicle model.
     */
    protected function processRow(array $row): ?Vehicle
    {
        if (empty($row)) {
            return null;
        }

        $skip = $this->skipImporting('import-map', $row, $this->import_map?->id);
        if ($skip) {
            return null;
        }

        $search_params = $this->importRowSearchParams($row);
        if ($search_params === null) {
            return null;
        }

        /** @var Vehicle $model */
        $model = Vehicle::query()->firstOrNew($search_params);

        $originalPrice = $model->price;
        $model->fill($this->getFillableData($row, $model));

        $model->is_published = $this->shouldBePublished(
            fn() => $model->is_published ?: false,
            'import-map',
            $model,
            $this->import_map?->id,
        );

        if ($model->isDirty()) {
            $model->save();
            $this->priceChangeEvents($model, 'import-map', $originalPrice);
        }

        $this->syncImages($model, $row['images'] ?? '');
        $this->syncFeatures($model, $row['features'] ?? '');
        $this->syncAttributes($model, $row);
        $this->syncVehicleAutotraderData($model, $row['autotrader_enabled'] ?? '');

        if ($model->wasRecentlyCreated) {
            Event::dispatch(new NewVehicleImported($model, $row, $this->getProviderName()));
        } else {
            Event::dispatch(new VehicleUpdatedFromImport($model, $row, $this->getProviderName()));
        }

        return $model;
    }

    /**
     * Get search parameters to find existing vehicle.
     */
    protected function importRowSearchParams(array $row): ?array
    {
        if (!empty($row['slug'])) {
            return ['slug' => $row['slug']];
        }

        if (empty($row['uuid'])) {
            if (empty($row['registration_number'])) {
                return null;
            }
            $row['uuid'] = 'import-' . $row['registration_number'];
        }

        $search_params = ['uuid' => $row['uuid']];

        if (!empty($row['stock_provider'])) {
            $search_params['stock_provider'] = $row['stock_provider'];
        }

        return $search_params;
    }

    /**
     * Get fillable data for the Vehicle model.
     */
    protected function getFillableData(array $vehicle, $model): array
    {
        if (empty(self::$vehicleFillable)) {
            self::$vehicleFillable = (new Vehicle())->getFillable();
        }

        if ($model->exists) {
            $tenantId = tenant('id');

            if (empty(self::$allowedFieldsToUpdate[$tenantId])) {
                if (isset($this->import_map)) {
                    self::$allowedFieldsToUpdate[$tenantId] = $this->import_map->fields
                        ->filter(fn($field) => $field->data['save_on_update'] ?? true)
                        ->pluck('on_model')
                        ->filter()
                        ->toArray();
                } else {
                    self::$allowedFieldsToUpdate[$tenantId] = self::$vehicleFillable;
                }
            }

            $vehicle = array_intersect_key($vehicle, array_flip(self::$allowedFieldsToUpdate[$tenantId]));
        }

        $vehicle['was_recently_synced'] = true;
        if (empty($vehicle['stock_provider'])) {
            $vehicle['stock_provider'] = $this->getProviderName();
        }

        return collect($vehicle)
            ->mapWithKeys(fn($value, $field) => [
                $this->getMappedField($field) => $this->getMappedValue($field, $value, $vehicle)
            ])
            ->only(self::$vehicleFillable)
            ->toArray();
    }

    /**
     * Ensure correct formatting for vehicle-specific fields.
     */
    protected function ensureFormattingForField(
        string $on_model,
        $value,
        array $row,
        ?string $valueCastType = null
    ): mixed {
        $country = Settings::get('app-details-country') ?? config('app.default_country');

        return match ($on_model) {
            'registration_number' => Country::normalizeNumberPlate($country, $value ?? '', false),
            'make_id' => $this->getMappedTaxonomy(TaxonomyMap::MAKE, $value, $row),
            'model_id' => $this->getMappedTaxonomy(TaxonomyMap::MODEL, $value, $row),
            'transmission_id' => $this->getMappedTaxonomy(TaxonomyMap::TRANSMISSION, $value, $row),
            'body_style_id' => $this->getMappedTaxonomy(TaxonomyMap::BODY_STYLE, $value, $row),
            'drivetrain_id' => $this->getMappedTaxonomy(TaxonomyMap::DRIVETRAIN, $value, $row),
            'fuel_type_id' => $this->getMappedTaxonomy(TaxonomyMap::FUEL_TYPE, $value, $row),
            'type' => match (strtolower($value ?? '')) {
                'lcv' => VehicleType::LCV->value,
                'motorcycle' => VehicleType::MOTORCYCLE->value,
                default => VehicleType::CAR->value,
            },
            'stock_arrival_date',
            'first_registration_date' => $valueCastType ? $value : $this->castToDateTime($value),
            'dealership_id' => $this->getDealershipId($value),
            default => is_string($value) ? trim($value) : $value,
        };
    }

    /**
     * Vehicle-specific row validation.
     */
    protected function rowFailsValidation(string $field, array $data): bool
    {
        if ($field === 'vin') {
            return strlen($data[$field]) < 17;
        }

        return false;
    }

    // =========================================================================
    // Vehicle-Specific Sync Methods
    // =========================================================================

    /**
     * Sync images for the vehicle.
     */
    protected function syncImages(Vehicle $vehicle, string $images): void
    {
        $this->syncImagesForModel($vehicle, $images);
    }

    /**
     * Sync features for the vehicle.
     */
    protected function syncFeatures(Vehicle $vehicle, string $features): void
    {
        $this->syncFeaturesForModel($vehicle, $features);
    }

    /**
     * Sync vehicle attributes from the import row.
     */
    protected function syncAttributes(Vehicle $vehicle, array $row): void
    {
        $attributes = collect($row)
            ->filter(fn($value, $key) => stripos($key, 'attribute.') !== false)
            ->mapWithKeys(function ($value, $key) use ($row) {
                $attribute_key = str_replace('attribute.', '', $key);

                return [
                    $attribute_key => $this->castValue($key, $value, $row),
                ];
            })
            ->toArray();

        $this->setVehicleAttributes($attributes, $vehicle);
    }

    /**
     * Sync AutoTrader publish settings.
     */
    protected function syncVehicleAutotraderData(Vehicle $vehicle, string $autotrader_enabled): void
    {
        if (!Settings::get('sales-channels-auto-trader-enabled')) {
            return;
        }

        $vehicle->autoTraderData()->updateOrCreate([
            'vehicle_id' => $vehicle->id
        ], [
            'publish_advert' => $this->castToBoolean($autotrader_enabled),
            'publish_profile' => $this->castToBoolean($autotrader_enabled),
        ]);
    }
}
