<?php

namespace App\Imports\Traits;

use App\TaxonomyMap;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Form;
use Mtc\MercuryDataModels\Franchise;
use Mtc\MercuryDataModels\OfferType;
use Mtc\MercuryDataModels\VehicleOffer;
use Mtc\MercuryDataModels\VehicleTechnicalData;

/**
 * Trait for importing VehicleOffers via ImportMap.
 * Uses BaseImportFlow for shared logic and adds offer-specific functionality.
 */
trait VehicleOfferImportFlow
{
    use BaseImportFlow;

    protected static array $offerFillable = [];
    protected ?Collection $franchises = null;
    protected ?Collection $offerTypes = null;

    /**
     * Load relationships needed for offer imports.
     */
    protected function loadRelationships(): void
    {
        $this->loadBaseRelationships();
        $this->franchises = Franchise::all()->pluck('id', 'name');
        $this->offerTypes = OfferType::all()->keyBy('slug');
    }

    /**
     * Process a single row into a VehicleOffer model.
     */
    protected function processRow(array $row): ?VehicleOffer
    {
        if (empty($row)) {
            return null;
        }

        $skip = $this->skipImporting('import-map', $row, $this->import_map?->id);
        if ($skip) {
            return null;
        }

        $search_params = $this->importRowSearchParams($row);
        if ($search_params === null) {
            return null;
        }

        // Handle deletion if marked
        if (!empty($row['to_be_deleted'])) {
            VehicleOffer::query()->where($search_params)->delete();
            return null;
        }

        /** @var VehicleOffer $model */
        $model = VehicleOffer::query()->firstOrNew($search_params);
        $model->fill($this->getFillableData($row, $model));

        if ($model->isDirty()) {
            $model->save();
        }

        $this->syncImages($model, $row['images'] ?? '');
        $this->syncFeatures($model, $row['features'] ?? '');
        $this->syncFinance($model, $row);
        $this->syncTechData($model, $row);


        // TODO: check if we need this...
        // mismatch between passed and expected type, however this was already in place...
        /*
        if ($model->wasRecentlyCreated) {
            Event::dispatch(new NewVehicleOfferImported($model, $row));
        }
        */

        return $model;
    }

    /**
     * Get search parameters to find existing offer.
     */
    protected function importRowSearchParams(array $row): ?array
    {
        if (!empty($row['slug'])) {
            return ['slug' => $row['slug']];
        }

        if (!empty($row['uuid'])) {
            return ['uuid' => $row['uuid']];
        }

        // Offers require either slug or uuid to identify
        return null;
    }

    /**
     * Get fillable data for the VehicleOffer model.
     */
    protected function getFillableData(array $offer, $model): array
    {
        if (empty(self::$offerFillable)) {
            self::$offerFillable = (new VehicleOffer())->getFillable();
        }

        if ($model->exists) {
            $tenantId = tenant('id');

            if (empty(self::$allowedFieldsToUpdate[$tenantId])) {
                if (isset($this->import_map)) {
                    self::$allowedFieldsToUpdate[$tenantId] = $this->import_map->fields
                        ->filter(fn($field) => $field->data['save_on_update'] ?? true)
                        ->pluck('on_model')
                        ->filter()
                        ->toArray();
                } else {
                    self::$allowedFieldsToUpdate[$tenantId] = self::$offerFillable;
                }
            }

            $offer = array_intersect_key($offer, array_flip(self::$allowedFieldsToUpdate[$tenantId]));
        }

        if (empty($offer['stock_provider'])) {
            $offer['stock_provider'] = $this->getProviderName();
        }

        // Transform array fields
        $this->transformColumnToArray('key_features', $offer);
        $this->transformColumnToArray('standard_spec', $offer);
        $this->transformColumnToArray('technical_spec', $offer);

        return collect($offer)
            ->mapWithKeys(fn($value, $field) => [
                $this->getMappedField($field) => $this->getMappedValue($field, $value, $offer)
            ])
            ->only(self::$offerFillable)
            ->toArray();
    }

    /**
     * Ensure correct formatting for offer-specific fields.
     */
    protected function ensureFormattingForField(
        string $on_model,
        $value,
        array $row,
        ?string $valueCastType = null
    ): mixed {
        return match ($on_model) {
            'make_id' => $this->getMappedTaxonomy(TaxonomyMap::MAKE, $value, $row),
            'model_id' => $this->getMappedTaxonomy(TaxonomyMap::MODEL, $value, $row),
            'transmission_id' => $this->getMappedTaxonomy(TaxonomyMap::TRANSMISSION, $value, $row),
            'body_style_id' => $this->getMappedTaxonomy(TaxonomyMap::BODY_STYLE, $value, $row),
            'drivetrain_id' => $this->getMappedTaxonomy(TaxonomyMap::DRIVETRAIN, $value, $row),
            'fuel_type_id' => $this->getMappedTaxonomy(TaxonomyMap::FUEL_TYPE, $value, $row),
            'franchise_id' => $this->getFranchiseId($value),
            'type_id' => $this->getOfferTypeId($value),
            'type' => $this->getOfferTypeSlug($value),
            'dealership_id' => $this->getDealershipId($value),
            'form_id' => $this->getEnquiryFormId(),
            'published_at',
            'unpublished_at' => $valueCastType ? $value : $this->castToDateTime($value),
            'new_car_type' => strtolower($value ?? ''),
            default => is_string($value) ? trim($value) : $value,
        };
    }

    /**
     * Override getMappedValue for offer-specific field handling.
     */
    protected function getMappedValue(string $field, $value, array $row)
    {
        return match ($field) {
            'make' => $value && !empty($this->makes[$value])
                ? $this->makes[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::MAKE, $value, $row),
            'model' => !empty($value) ? $this->getModelValue($value, $row) : null,
            'dealership' => $this->getDealershipId($value),
            'fuel_type' => $value && !empty($this->fuel_types[$value])
                ? $this->fuel_types[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::FUEL_TYPE, $value, $row),
            'body_style' => $value && !empty($this->body_styles[$value])
                ? $this->body_styles[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::BODY_STYLE, $value, $row),
            'drivetrain' => $value && !empty($this->drivetrains[$value])
                ? $this->drivetrains[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::DRIVETRAIN, $value, $row),
            'transmission' => $value && !empty($this->transmissions[$value])
                ? $this->transmissions[$value]
                : $this->getMappedTaxonomy(TaxonomyMap::TRANSMISSION, $value, $row),
            'franchise' => $this->getFranchiseId($value),
            'featured', 'published' => $this->castToBoolean($value),
            'price', 'deposit', 'full_price',
            'battery_range', 'battery_capacity_kwh', 'battery_usable_capacity_kwh',
            'battery_charge_time', 'battery_quick_charge_time', 'battery_quick_charge_level',
            'engine_size_cc', 'mpg', 'co2' => $this->sanitiseNumber($value),
            default => $this->castValue($field, $value, $row),
        };
    }

    /**
     * Override getMappedField for offer-specific mappings.
     */
    protected function getMappedField(string $field): string
    {
        return match ($field) {
            'make' => 'make_id',
            'model' => 'model_id',
            'dealership' => 'dealership_id',
            'fuel_type' => 'fuel_type_id',
            'body_style' => 'body_style_id',
            'drivetrain' => 'drivetrain_id',
            'transmission' => 'transmission_id',
            'franchise' => 'franchise_id',
            default => $field,
        };
    }

    // =========================================================================
    // Offer-Specific Helper Methods
    // =========================================================================

    protected function getFranchiseId(?string $franchise_name): ?int
    {
        if (empty($franchise_name)) {
            return null;
        }
        return $this->franchises[$franchise_name] ?? null;
    }

    protected function getOfferTypeId(?string $type): ?int
    {
        if (empty($type)) {
            return null;
        }
        return $this->offerTypes->get(strtolower($type))?->id;
    }

    protected function getOfferTypeSlug(?string $type): ?string
    {
        if (empty($type)) {
            return null;
        }
        return $this->offerTypes->get(strtolower($type))?->slug;
    }

    protected function getEnquiryFormId(): ?int
    {
        return Form::query()
            ->where('name', '=', 'Offer Enquiry')
            ->where('is_active', '=', true)
            ->first()?->id;
    }

    protected function transformColumnToArray(string $column, array &$row, string $separator = ','): void
    {
        if (array_key_exists($column, $row) && is_string($row[$column])) {
            $row[$column] = collect(explode($separator, $row[$column]))
                ->map(fn(string $item) => trim($item))
                ->filter(fn($item) => !empty($item))
                ->values()
                ->toArray();
        }
    }

    protected function sanitiseNumber($value): ?float
    {
        if (empty($value)) {
            return is_numeric($value) ? 0 : null;
        }

        if (is_numeric($value)) {
            return (float) $value;
        }

        return (float) str_replace(['£', ',', '%'], '', $value);
    }

    // =========================================================================
    // Offer-Specific Sync Methods
    // =========================================================================

    /**
     * Sync images for the offer.
     */
    protected function syncImages(VehicleOffer $offer, string $images): void
    {
        $this->syncImagesForModel($offer, $images);
    }

    /**
     * Sync features for the offer.
     */
    protected function syncFeatures(VehicleOffer $offer, string $features): void
    {
        $this->syncFeaturesForModel($offer, $features);
    }

    /**
     * Sync finance data for the offer.
     */
    protected function syncFinance(VehicleOffer $offer, array $row): void
    {
        $data = collect([
            'finance_type' => $row['finance_type'] ?? null,
            'first_payment' => $this->sanitiseNumber($row['first_payment'] ?? null),
            'monthly_price' => $this->sanitiseNumber($row['monthly_price'] ?? null),
            'final_payment' => $this->sanitiseNumber($row['final_payment'] ?? null),
            'full_price' => $this->sanitiseNumber($row['full_price'] ?? null),
            'deposit' => $this->sanitiseNumber($row['deposit'] ?? null),
            'total_amount' => $this->sanitiseNumber($row['total_amount'] ?? null),
            'total_credit_amount' => $this->sanitiseNumber($row['total_credit_amount'] ?? null),
            'apr' => $this->sanitisePercentage($row['apr'] ?? null),
            'interest_rate' => $this->sanitisePercentage($row['interest_rate'] ?? null),
            'number_of_payments' => $this->sanitiseNumber($row['number_of_payments'] ?? null),
            'term' => $this->sanitiseNumber($row['term'] ?? null),
            'annual_mileage' => $this->sanitiseNumber($row['annual_mileage'] ?? null),
            'customer_deposit' => $this->sanitiseNumber($row['customer_deposit'] ?? null),
            'dealer_deposit_contribution' => $this->sanitiseNumber($row['dealer_deposit_contribution'] ?? null),
            'option_to_purchase_fee' => $this->sanitiseNumber($row['option_to_purchase_fee'] ?? null),
            'excess_mileage_charge' => $this->sanitiseNumber($row['excess_mileage_charge'] ?? null),
        ])->filter()->toArray();

        // Only save if any finance details are set
        if (!empty($data)) {
            $finance = $offer->finance()->firstOrNew(['provider' => 'import']);
            $finance->fill($data)->save();
        }
    }

    protected function sanitisePercentage($value): ?float
    {
        if (empty($value)) {
            return null;
        }

        $value = str_replace([',', '%'], ['.', ''], $value);

        if (!is_numeric($value)) {
            return null;
        }

        return (float) $value;
    }

    /**
     * Sync technical data for the offer.
     * Handles offer_tech_data.* fields from the import row.
     */
    protected function syncTechData(VehicleOffer $offer, array $row): void
    {
        $techData = collect($row)
            ->filter(fn($value, $key) => str_starts_with($key, 'offer_tech_data.'))
            ->mapWithKeys(function ($value, $key) {
                $description = str_replace('offer_tech_data.', '', $key);
                return [$description => $value];
            })
            ->filter(fn($value) => !empty($value));

        if ($techData->isEmpty()) {
            return;
        }

        $techData->each(function ($value, $description) use ($offer) {
            VehicleTechnicalData::query()->updateOrCreate([
                'vehicle_id' => $offer->id,
                'vehicle_type' => 'offer',
                'description' => $description,
            ], [
                'value' => $value,
                'code' => '',
                'category' => '',
            ]);
        });
    }
}
