<?php

namespace App\Imports;

use App\Imports\Traits\VehicleOfferImportFlow;
use App\Traits\MapsTaxonomies;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithStartRow;

/**
 * Import class for VehicleOffers using ImportMap field mappings.
 */
class VehicleOfferImport implements ToCollection, WithHeadingRow, WithStartRow
{
    use DispatchesJobs;
    use MapsTaxonomies;
    use VehicleOfferImportFlow;

    private array $additional_tasks = [];

    public function __construct(
        private readonly ?string $importType = '',
    ) {
        $this->loadRelationships();
    }

    public function model(array $source)
    {
        $row = match ($this->importType) {
            'default' => $source,
            default => $this->mapBasedOnImportMap($source)
        };

        try {
            $model = $this->processRow($row);
            if ($model) {
                $this->storeUnmappedTaxonomy($model);
            }
            return $model;
        } catch (\Exception $exception) {
            Log::error(tenant('id') . ' - ' . $exception->getMessage(), [
                'data' => $source,
            ]);
            return null;
        }
    }

    public function startRow(): int
    {
        return 2;
    }

    public function headingRow(): int
    {
        return 1;
    }

    public function validateHeadings(array $headings): void
    {
        $missing = array_diff($this->getExpectedHeadings(), $headings);
        if (!empty($missing)) {
            throw new \Exception('Columns not found in file: ' . implode(', ', $missing));
        }
    }

    protected function getDetailsForTaxonomyMap(array $record): array
    {
        return $record;
    }

    public function collection(Collection $collection): void
    {



        $this->triggerAdditionalTasks(
            $collection->map(fn(Collection $row) => $this->model($row->toArray()))->filter()
        );
    }

    private function triggerAdditionalTasks(Collection $offers): void
    {
        foreach ($this->additional_tasks as $task) {
            match ($task) {
                // Add offer-specific additional tasks here as needed
                // 'offer-specs' => FetchOfferSpecData::dispatch($offers),
                default => null,
            };
        }
    }

    private function getExpectedHeadings(): array
    {
        if ($this->importType === 'default') {
            return []; // No default export template for offers yet
        }

        return $this->import_map->fields
            ->filter(fn($field) => empty($field['data']['is_static']))
            ->pluck('on_file')
            ->toArray();
    }
}
