<?php

namespace App\Traits;

use App\Events\VehiclePriceChangedDuringStockSync;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Event;
use Mtc\ContentManager\Models\Media;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;

trait StockSyncTraits
{
    use MapsTaxonomies;

    /**
     * Get the model class to use for stock sync operations.
     * Defaults to Vehicle unless import_map specifies vehicle_offer type.
     */
    private function getStockSyncModelClass(): string
    {
        if (isset($this->import_map) && $this->import_map->type === 'vehicle_offer') {
            return VehicleOffer::class;
        }

        return Vehicle::class;
    }

    /**
     * Find all stock locations to fetch data from
     * These are stored against dealerships
     *
     * @return Collection
     */
    private function getAllStockLocations(string $provider_attribute): Collection
    {
        $this->dealerships = Dealership::all()
            ->each(function (Dealership $dealership) use ($provider_attribute) {
                $dealership->stock_location = $dealership->data[$provider_attribute] ?? null;
            });

        return $this->dealerships
            ->map(fn(Dealership $dealership) => $dealership->data[$provider_attribute] ?? null)
            ->filter();
    }

    /**
     * Get the dealership id from stock location
     *
     * @param $locationId
     * @return string|null
     */
    private function dealershipId($locationId): ?string
    {
        return $this->dealerships
            ->where('stock_location', $locationId)
            ->first()
            ?->id;
    }

    private function priceChangeEvents(Vehicle $vehicle, string $provider, mixed $originalPrice = null): void
    {
        if (!$vehicle->wasChanged('price')) {
            return;
        }

        // If original price provided, do numeric comparison to avoid false positives
        // from type differences (e.g., '10000' string vs 10000.00 decimal)
        if ($originalPrice !== null && (float) $originalPrice === (float) $vehicle->price) {
            return;
        }

        Event::dispatch(new VehiclePriceChangedDuringStockSync($vehicle, $provider));
    }

    private function removeOld(Collection $allRecords): void
    {
        $modelClass = $this->getStockSyncModelClass();
        $modelClass::query()
            ->where('stock_provider', self::NAME)
            ->whereNotIn('uuid', $allRecords)
            ->delete();
    }

    private function removeNotSynced(bool $check_pending = true): void
    {
        // Sync tracking only supported for Vehicle imports
        if ($this->getStockSyncModelClass() !== Vehicle::class) {
            return;
        }

        if ($this->enoughRecordsSyncedRecently(config('automotive.sync_margin_for_removing'), $check_pending)) {
            $query = Vehicle::query()
                ->where('stock_provider', $this->getProviderName())
                ->where('was_recently_synced', 0);

            if ($check_pending) {
                $query->whereNull('pending_stock_sync');
            }

            $query->delete();
        }
    }

    private function markVehiclesToBeSynced(): void
    {
        // Sync tracking only supported for Vehicle imports
        if ($this->getStockSyncModelClass() !== Vehicle::class) {
            return;
        }

        Vehicle::query()
            ->where('stock_provider', $this->getProviderName())
            ->update([
                'was_recently_synced' => false,
            ]);
    }

    /**
     * @deprecated Use enoughRecordsSyncedRecently instead
     */
    private function enoughVehiclesSyncedRecently(float $margin, bool $check_pending): bool
    {
        return $this->enoughRecordsSyncedRecently($margin, $check_pending);
    }

    private function enoughRecordsSyncedRecently(float $margin, bool $check_pending): bool
    {
        // Sync tracking only supported for Vehicle imports
        if ($this->getStockSyncModelClass() !== Vehicle::class) {
            return true;
        }

        $syncedQuery = Vehicle::query()
            ->where('stock_provider', $this->getProviderName())
            ->where('was_recently_synced', 1);

        $totalQuery = Vehicle::query()
            ->where('stock_provider', $this->getProviderName());

        if ($check_pending) {
            $syncedQuery->whereNull('pending_stock_sync');
            $totalQuery->whereNull('pending_stock_sync');
        }

        $synced = $syncedQuery->count();
        $provider_total = $totalQuery->count();

        return empty($provider_total) ?: ($synced / $provider_total) > $margin;
    }

    private function removeWithMedia(Model $model): void
    {
        $ids = $model->mediaUses()->pluck('media_id');
        $model->mediaUses()->delete();
        $model->delete();
        Media::query()->whereIn('id', $ids)->whereDoesntHave('uses')->delete();
    }
}
