<?php

namespace Tests\Tenant;

use App\Facades\Settings;
use App\Jobs\ImportImagesFromUrlList;
use App\Modules\Stock\Skupenet;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;
use Tests\TestCase;
use App\VehicleType;

class SkupenetTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;
    public function testNameAndDealershipFields()
    {
        $this->assertEquals('Skupenet', (new Skupenet())->name());
        $this->assertArrayHasKey('skupenet-location-id', (new Skupenet())->dealershipAdditionalDataFields());
    }

    public function testImport()
    {
        Queue::fake();
        Http::fake([
            'https://feeds.dealerhub.ie/api/v1/stock?*' => Http::response($this->response()),
            "*" => Http::response([], 400),
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'foo',
            'group' => 'foo',
            'name' => 'foo',
            'config_key' => 'app-details-currency',
            'value' => 'GBP',
            'type' => 'string'
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'foo',
            'group' => 'foo',
            'name' => 'foo',
            'config_key' => 'stock-skupenet-sync-images',
            'value' => true,
            'type' => 'boolean'
        ]);

        Dealership::factory()->create(['data' => ['skupenet-location-id' => 123]]);
        (new Skupenet())->runScheduledImport();

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://feeds.dealerhub.ie/api/v1/stock?dealerIds=123&pageSize=50&page=1';
        });
        $this->assertEquals(4, Vehicle::query()->count());
        $this->assertTrue(Vehicle::query()->where('vrm_condensed', '212D1122')->exists());

        $this->assertEquals(
            VehicleType::CAR->value,
            Vehicle::where('uuid', 123)->first()->type
        );

        $this->assertEquals(
            VehicleType::LCV->value,
            Vehicle::where('uuid', 333)->first()->type
        );

        $this->assertEquals(
            VehicleType::MOTORCYCLE->value,
            Vehicle::where('uuid', 444)->first()->type
        );

        $v123 = Vehicle::where('uuid', 123)->first();
        $this->assertNotNull($v123);
        $this->assertEquals(250, (int) $v123->battery_range);
        $this->assertEquals(44, (int) $v123->battery_capacity_kwh);
        // reg date should be extracted from VRM
        $this->assertEquals('2021-07-01', $v123->first_registration_date->format('Y-m-d'));

        $v222 = Vehicle::where('uuid', 222)->first();
        $this->assertNotNull($v222);
        $this->assertNull($v222->battery_range);
        $this->assertNull($v222->battery_capacity_kwh);
        // reg date cannot be extracted from vrm and no manufacture year to guess from
        $this->assertNull($v222->first_registration_date);

        $v333 = Vehicle::where('uuid', 333)->first();
        // reg date should be guessed from manufacture year
        $this->assertEquals('2019-01-01', $v333->first_registration_date->format('Y-m-d'));

        Queue::assertPushed(ImportImagesFromUrlList::class);
    }

    public function testInvalidRegDateFromVrm()
    {
        $skupenet = new Skupenet();

        // non-Irish and non-UK VRM format
        $this->assertNull($skupenet->getRegistrationDateFromRegistrationNumber('12345'));

        // valid UK format
        $this->assertEquals('2023-03-01', $skupenet->getRegistrationDateFromRegistrationNumber('AB23XYZ'));

        // valid Irish format
        $this->assertEquals('2021-01-01', $skupenet->getRegistrationDateFromRegistrationNumber('211D3008'));

        // Northern Ireland format
        $this->assertNull($skupenet->getRegistrationDateFromRegistrationNumber('ABC1234'));
    }

    private function response(): array
    {
        return [
            'stock' => [
                [
                    'id' => 123,
                    'dhfDealerId' => 123,
                    'price' => 11223,
                    'currency' => 'GBP',
                    'stockType' => 'CARS',
                    'description' => 'lorem ipsum dolor sit amet',
                    'attributes' => [
                        'registration' => '212D1122',
                        'make' => 'Volkswagen',
                        'model' => 'ID.5',
                        'trim' => 'MAX',
                        'bodyType' => 'Sedan',
                        'batteryCapacityKwh' => 44.5,
                        'batteryRangeKm' => 250.4,

                    ],
                    'images' => [
                        [
                            'url' => 'a.com/123',
                        ],
                        [
                            'url' => 'a.com/222',
                        ],
                    ],
                ],
                [
                    'id' => 222,
                    'dhfDealerId' => 123,
                    'price' => 13443,
                    'currency' => 'GBP',
                    'stockType' => 'CARS',
                    'description' => 'lorem ipsum dolor sit amet',
                    'attributes' => [
                        'registration' => '12345',
                        'make' => 'Ford',
                        'model' => 'Focus',
                        'bodyType' => 'Sedan',
                    ],
                ],
                [
                    'id' => 333,
                    'dhfDealerId' => 123,
                    'price' => 13443,
                    'currency' => 'GBP',
                    'stockType' => 'COMMERCIALS',
                    'description' => 'lorem ipsum dolor sit amet',
                    'attributes' => [
                        'registration' => '45678',
                        'make' => 'Ford',
                        'model' => 'Transit',
                        'bodyType' => 'Commercial/Van',
                        'year' => '2019',
                    ],
                ],
                [
                    'id' => 444,
                    'dhfDealerId' => 123,
                    'price' => 900,
                    'currency' => 'GBP',
                    'stockType' => 'bike',
                    'description' => 'test bike',
                    'attributes' => [
                        'registration' => 'BIKE123',
                        'make' => 'Yamaha',
                        'model' => 'MT-07',
                    ],
                ],
            ],
            'paging' => [
                'totalPages' => 1,
            ],
        ];
    }
}
