<?php

namespace Tests\Tenant;

use App\Facades\Settings;
use Carbon\Carbon;
use Database\Seeders\Tenant\FinanceSettingSeeder;
use Database\Seeders\Tenant\SantanderFinanceSettingSeeder;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Http;
use Mtc\MercuryDataModels\Finance\Config\SantanderConfig;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;
use Mtc\MercuryDataModels\Finance\Services\SantanderFinance;
use Tests\TenantTestCase;

class SantanderFinanceServiceTest extends TenantTestCase
{
    protected function seedAndConfigureSantander(string $environment = 'uat'): void
    {
        $this->seed(FinanceSettingSeeder::class);
        $this->seed(SantanderFinanceSettingSeeder::class);

        // Disable other providers to ensure Santander is selected
        Settings::update('finance-codeweavers-enabled', false);
        Settings::update('finance-ivendi-enabled', false);
        Settings::update('finance-evolution-enabled', false);

        Settings::update('finance-santander-enabled', true);
        Settings::update('finance-santander-username', 'test-user');
        Settings::update('finance-santander-password', 'test-pass');
        Settings::update('finance-santander-dealer-code', 'TEST123');
        Settings::update('finance-santander-environment', $environment);
    }

    protected function makeFinanceRequest(): FinanceRequestData
    {
        return new FinanceRequestData(
            uuid: 'abc123',
            registration_number: 'SD12 ABC',
            cap_id: 1231122,
            dealer_id: 'abc123',
            engine_size: 1999,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 48,
            mileage: 100000,
            annual_mileage: 10000,
            price: 25000,
            deposit: 5000,
            credit_rating: 'excellent',
        );
    }

    public function testGoodRequestCalculation()
    {
        Http::fake([
            'https://ficresuktestcert.santanderconsumer.com/*' => Http::response($this->successResponse()),
            '*' => Http::response([], 400),
        ]);

        $this->seedAndConfigureSantander();

        $provider = FinanceServiceHelper::initializeForSite();

        $this->assertInstanceOf(SantanderFinance::class, $provider);

        $result = $provider->calculate($this->makeFinanceRequest());

        Http::assertSent(function (Request $request) {
            return str_contains($request->url(), 'ficresuktestcert.santanderconsumer.com/QuotingTool/api/multi')
                && $request['ThirdPartyUser'] === 'test-user'
                && $request['ThirdPartyPassword'] === 'test-pass'
                && isset($request['QuoteRequestList'])
                && $request['QuoteRequestList'][0]['DealerCode'] === 'TEST123'
                && $request['QuoteRequestList'][0]['CapCode'] == 1231122
                && $request['QuoteRequestList'][0]['CostPrice'] == 25000
                && $request['QuoteRequestList'][0]['Period'] == 48
                && $request['QuoteRequestList'][0]['Deposit'] == 5000;
        });

        $this->assertEquals(1, $result->count());
        $finance = $result->first();
        $this->assertEquals(350.00, $finance->monthly_price);
        $this->assertEquals(48, $finance->term);
        $this->assertEquals(9.9, $finance->apr);
        $this->assertEquals(5000, $finance->total_deposit);
        $this->assertEquals(25000, $finance->cash_price);
        $this->assertEquals('santander', $finance->provider);
    }

    public function testLiveEnvironmentUrl()
    {
        Http::fake([
            'https://ficresuk.santanderconsumer.com/*' => Http::response($this->successResponse()),
            '*' => Http::response([], 400),
        ]);

        $this->seedAndConfigureSantander('live');

        $provider = FinanceServiceHelper::initializeForSite();
        $provider->calculate($this->makeFinanceRequest());

        Http::assertSent(function (Request $request) {
            return str_contains($request->url(), 'ficresuk.santanderconsumer.com/QuotingTool/api/multi');
        });
    }

    public function testServiceUnavailable()
    {
        Http::fake([
            'https://ficresuktestcert.santanderconsumer.com/*' => Http::response([], 500),
            '*' => Http::response([], 400),
        ]);

        $this->seedAndConfigureSantander();

        $provider = FinanceServiceHelper::initializeForSite();
        $result = $provider->calculate($this->makeFinanceRequest());

        $this->assertEmpty($result);
    }

    public function testNoQuotesAvailable()
    {
        Http::fake([
            'https://ficresuktestcert.santanderconsumer.com/*' => Http::response($this->emptyResponse()),
            '*' => Http::response([], 400),
        ]);

        $this->seedAndConfigureSantander();

        $provider = FinanceServiceHelper::initializeForSite();
        $result = $provider->calculate($this->makeFinanceRequest());

        $this->assertEmpty($result);
    }

    public function testQuoteStatusNotSuccessful()
    {
        Http::fake([
            'https://ficresuktestcert.santanderconsumer.com/*' => Http::response($this->failedQuoteResponse()),
            '*' => Http::response([], 400),
        ]);

        $this->seedAndConfigureSantander();

        $provider = FinanceServiceHelper::initializeForSite();
        $result = $provider->calculate($this->makeFinanceRequest());

        $this->assertEmpty($result);
    }

    public function testConfigReadsSettings()
    {
        $this->seed(SantanderFinanceSettingSeeder::class);
        Settings::update('finance-santander-enabled', true);
        Settings::update('finance-santander-username', 'my-username');
        Settings::update('finance-santander-password', 'my-password');
        Settings::update('finance-santander-dealer-code', 'DEALER001');
        Settings::update('finance-santander-environment', 'live');

        $config = new SantanderConfig();

        $this->assertTrue($config->enabled());
        $this->assertEquals('my-username', $config->username());
        $this->assertEquals('my-password', $config->password());
        $this->assertEquals('DEALER001', $config->dealerCode());
        $this->assertEquals('live', $config->environment());
        $this->assertEquals('https://ficresuk.santanderconsumer.com', $config->baseUrl());
    }

    public function testConfigUatEnvironment()
    {
        $this->seed(SantanderFinanceSettingSeeder::class);
        Settings::update('finance-santander-environment', 'uat');

        $config = new SantanderConfig();

        $this->assertEquals('https://ficresuktestcert.santanderconsumer.com', $config->baseUrl());
    }

    public function testHpAndPcpProductIds()
    {
        Http::fake([
            'https://ficresuktestcert.santanderconsumer.com/*' => Http::response($this->successResponse()),
            '*' => Http::response([], 400),
        ]);

        $this->seedAndConfigureSantander();
        Settings::update('finance-santander-hp-product-id', 'HP001');
        Settings::update('finance-santander-pcp-product-id', 'PCP001');

        $provider = FinanceServiceHelper::initializeForSite();
        $result = $provider->calculate($this->makeFinanceRequest());

        // Should have made 2 requests (one for HP, one for PCP)
        Http::assertSentCount(2);

        // Should return 2 results (HP and PCP)
        $this->assertEquals(2, $result->count());

        // Verify HP result has correct finance type
        $hpResult = $result->get('HP');
        $this->assertNotNull($hpResult);
        $this->assertEquals('Santander HP', $hpResult->product_name);

        // Verify PCP result has correct finance type
        $pcpResult = $result->get('PCP');
        $this->assertNotNull($pcpResult);
        $this->assertEquals('Santander PCP', $pcpResult->product_name);
    }

    public function testOnlyHpProductId()
    {
        Http::fake([
            'https://ficresuktestcert.santanderconsumer.com/*' => Http::response($this->successResponse()),
            '*' => Http::response([], 400),
        ]);

        $this->seedAndConfigureSantander();
        Settings::update('finance-santander-hp-product-id', 'HP001');

        $provider = FinanceServiceHelper::initializeForSite();
        $result = $provider->calculate($this->makeFinanceRequest());

        // Verify ProductId is sent in request
        Http::assertSent(function (Request $request) {
            return $request['QuoteRequestList'][0]['ProductId'] === 'HP001';
        });

        $this->assertEquals(1, $result->count());
        $this->assertEquals('Santander HP', $result->first()->product_name);
    }

    private function successResponse(): array
    {
        return [
            'quoteResponseList' => [
                [
                    'quoteId' => 1,
                    'quoteStatus' => 1,
                    'period' => 48,
                    'apr' => 9.9,
                    'rate' => 8.5,
                    'deposit' => 5000,
                    'costPrice' => 25000,
                    'totalAmount' => 21800,
                    'ballon' => 8000,
                    'paf' => 10,
                    'bff' => 0,
                    'payments' => [
                        [
                            'months' => 1,
                            'value' => 350.00,
                        ],
                        [
                            'months' => 47,
                            'value' => 350.00,
                        ],
                    ],
                ],
            ],
        ];
    }

    private function emptyResponse(): array
    {
        return [
            'quoteResponseList' => [],
        ];
    }

    private function failedQuoteResponse(): array
    {
        return [
            'quoteResponseList' => [
                [
                    'quoteId' => 1,
                    'quoteStatus' => 0,
                    'errorMessage' => 'Vehicle not eligible for finance',
                ],
            ],
        ];
    }
}
