<?php

namespace Tests\Unit;

use App\Facades\Settings;
use App\Mail\NewEnquiryMail;
use Carbon\Carbon;
use Database\Seeders\Global\CountrySeeder;
use Database\Seeders\Tenant\RapidRTCSettingSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\EnquiryAction;
use Mtc\MercuryDataModels\Dealership;
use Tests\TenantTestCase;
use Tests\TestCase;
use Tests\UserForTenant;

class SendToRapidRTCTest extends TenantTestCase
{

    public function testSendEnquiry()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'failed_at' => null,
            'processed_at' => null,
            'data' => [
                'fallback-recipient' => 'john.doe@mtc.co.uk',
            ]
        ]);
        Mail::fake();
        Enquiries::makeAction('send-to-rapid-rtc')->handle($action, $enquiry);

        Mail::assertSent(config('crm.new_enquiry_email'), function (NewEnquiryMail $mail) {
            return $mail->hasTo('john.doe@mtc.co.uk');
        });
    }

    public function testSkipsAlreadyProcessed()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'processed_at' => Carbon::now(),
            'data' => [
                'fallback-recipient' => 'john.doe@mtc.co.uk',
            ]
        ]);
        Mail::fake();
        Enquiries::makeAction('send-to-rapid-rtc')->handle($action, $enquiry);

        Mail::assertNotSent(config('crm.new_enquiry_email'));
    }

    public function testSkipsTooManyAttempts()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'attempts' => 3,
            'data' => [
                'fallback-recipient' => 'john.doe@mtc.co.uk',
            ]
        ]);
        Mail::fake();
        Enquiries::makeAction('send-to-rapid-rtc')->handle($action, $enquiry);

        Mail::assertNotSent(NewEnquiryMail::class);
    }

    public function testGetFormFields()
    {
        $fields = Enquiries::makeAction('send-to-rapid-rtc')->formAdditionalDataFields();

        $this->assertIsArray($fields);
        $this->assertArrayHasKey('fallback-recipient', $fields);
    }

    public function testGetDealershipFields()
    {
        $fields = Enquiries::makeAction('send-to-rapid-rtc')->dealershipAdditionalDataFields();

        $this->assertIsArray($fields);
        $this->assertArrayHasKey('rapid-rtc-mailbox', $fields);
    }

    public function testGetErrorReason()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'processed_at' => null,
            'failed_at' => null,
            'data' => [
                'fallback-recipient' => '',
            ]
        ]);
        Mail::fake();
        $handler = Enquiries::makeAction('send-to-rapid-rtc');
        $this->assertFalse($handler->handle($action, $enquiry));
        $this->assertIsString($handler->failureReason());
    }
}
