<?php

namespace Tests\Feature;

use App\Facades\Settings;
use App\Jobs\BulkImportJob;
use Database\Seeders\MercuryTheme\ContentElementSeeder;
use Database\Seeders\MercuryTheme\TemplateSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Queue;
use Mtc\ContentManager\PageStatus;
use Mtc\MercuryDataModels\Template;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;
use Mtc\MercuryDataModels\VehicleOfferContentHistory;
use Tests\TestCase;
use Tests\UserForTenant;

class VehicleOfferControllerTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    protected $tenancy = true;

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        VehicleOffer::factory(5)->create();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-offers.index'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertCount(5, $response->json('data'));
    }

    public function testStore()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.store'), [
                'name' => 'foo',
            ]);

        $response->assertStatus(201);
        $this->assertEquals('foo', $response->json('offer.name'));
        $this->assertTrue(VehicleOffer::query()->where('name', 'foo')->exists());

    }

    public function testShow()
    {
        $offer = VehicleOffer::factory()->create([
            'name' => 'foo',
            'type_id' => 11,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-offers.show', $offer));

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('offer.name'));
        $this->assertEquals(11, $response->json('offer.type_id'));
    }

    public function testUpdate()
    {
        $offer = VehicleOffer::factory()->create([
            'name' => 'foo',
            'type_id' => 11,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.vehicle-offers.update', $offer), [
                'name' => 'baz',
                'featured' => 0,
                'published' => 1,
                'derivative' => 'foo baz bar',
                'make_id' => 8,
                'model_id' => 9,
                'type_id' => 10,
                'form_id' => 11,
                'seo' => [],
                'data' => [],
                'content' => [],
            ]);

        $offer->refresh();
        $response->assertStatus(200);
        $this->assertEquals('baz', $response->json('offer.name'));
        $this->assertEquals(10, $response->json('offer.type_id'));
        $this->assertEquals('baz', $offer->name);
        $this->assertEquals('foo baz bar', $offer->derivative);
        $this->assertEquals(8, $offer->make_id);
        $this->assertEquals(9, $offer->model_id);
        $this->assertEquals(10, $offer->type_id);
        $this->assertEquals(11, $offer->form_id);
        $this->assertEquals(1, $offer->published);
    }

    public function testDestroy()
    {
        $offer = VehicleOffer::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->deleteJson(route('tenant.vehicle-offers.destroy', $offer));

        $response->assertStatus(200);
        $this->assertFalse(VehicleOffer::query()->where('id', $offer->id)->exists());
    }

    public function testGetFeatured()
    {
        VehicleOffer::factory(3)->create(['featured' => 1]);
        VehicleOffer::factory(5)->create(['featured' => 0]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-offers.get-featured'));
        $response->assertStatus(200);

        $this->assertIsArray($response->json('data'));
        $this->assertCount(3, $response->json('data'));
    }

    public function testSetFeatured()
    {
        $to_set_featured = VehicleOffer::factory(3)->create(['featured' => 0]);
        VehicleOffer::factory(5)->create(['featured' => 0]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.vehicle-offers.set-featured'), [
                'offer_ids' => $to_set_featured->pluck('id')
            ]);

        $response->assertStatus(200);

        $this->assertIsArray($response->json('data'));
        $this->assertCount(3, $response->json('data'));
        $this->assertEquals(3, VehicleOffer::query()->where('featured', 1)->count());

        $featured_ids = collect($response->json('data'))->pluck('id');
        $this->assertEquals($to_set_featured->pluck('id')->toArray(), $featured_ids->toArray());
    }

    public function testExport()
    {
        VehicleOffer::factory(3)->create(['featured' => 1]);
        VehicleOffer::factory(5)->create(['featured' => 0]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.export'));
        $response->assertStatus(200);
        $response->assertDownload();
    }

    public function testTemplateSync()
    {
        $template = Template::factory()->create();
        $offer = VehicleOffer::factory()->create(['template_id' => $template->id]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.sync-template', $offer));
        $response->assertStatus(200);
    }

    public function testVersionRestore()
    {
        $offer = VehicleOffer::factory()->create();
        $version = VehicleOfferContentHistory::factory()->create(['offer_id' => $offer->id]);


        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.restore-version', [$offer, $version]));
        $response->assertStatus(200);
        $version->refresh();
        $this->assertTrue($version->is_active);
    }

    public function testImport()
    {
        Queue::fake();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.import'), [
                'file' => UploadedFile::fake()->create('offers.xlsx'),
            ]);

        $response->assertStatus(200);
        Queue::assertPushed(BulkImportJob::class);
    }

    public function testVehicleImport()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.import'), [
                'file' => UploadedFile::fake()
                    ->createWithContent('offers.xlsx', File::get(dirname(__DIR__) . '/data/offers.xlsx')),
            ]);

        $response->assertStatus(200);
        $this->assertEquals(14, VehicleOffer::query()->count());
    }

    public function testContentUpdateAdVersionSaving()
    {
        tenant()->tier = 'pro';
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'value' => true,
            'type' => 'bool',
            'config_key' => 'offers-versioning-enabled',
        ]);
        $this->seed(ContentElementSeeder::class);
        $this->seed(TemplateSeeder::class);
        $template = Template::query()->where('slug', 'offers-list')->first();
        $offer = VehicleOffer::factory()->create([
            'name' => 'foo',
            'template_id'  => $template->id,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.sync-template', $offer));

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-offers.show', $offer));

        $newCar = $response->json('offer');

        $newCar['content'][0]['children'][0]['content'] = 'foo';
        $newCar['content'][0]['children'][1]['content'] = 'baz bar';
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.vehicle-offers.update', $offer), $newCar);

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('offer.content.0.children.0.content'));
        $this->assertEquals('baz bar', $response->json('offer.content.0.children.1.content'));
        $this->assertEquals(1, $offer->versions()->count());
    }

    public function testCopy()
    {
        $offer = VehicleOffer::factory()->create([
            'name' => 'foo',
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-offers.copy', $offer));

        $response->assertStatus(201);
        $this->assertTrue(VehicleOffer::query()->where('id', $response->json('offer.id'))->exists());
        $this->assertNotEquals($offer->id, $response->json('offer.id'));
        $this->assertEquals($offer->name, $response->json('offer.name'));
    }
}
