<?php

namespace App\Exports;

use App\Facades\Settings;
use App\Facades\Site;
use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class CarSnipExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::CAR_SNIP),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::CAR_SNIP)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'Feed_Id' => $vehicle->dealership?->id,
                    'Vehicle_Id' => $vehicle->uuid,
                    'Cap_Id' => $vehicle->cap_id,
                    'Make' => $vehicle->make?->name,
                    'Model' => $vehicle->model?->name,
                    'Variant' => $vehicle->derivative,
                    'Year' => $vehicle->manufacture_year,
                    'Price' => $vehicle->price,
                    'FullRegistration' => $vehicle->registration_number,
                    'Mileage' => $vehicle->odometer_mi,
                    'images' => $vehicle->mediaUses
                        ->filter(fn($mediaUse) => $mediaUse->media)
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large'))
                        ->implode(','),
                    'Vehicle_URL' => Site::vehicleUrl($vehicle),
                    'Colour' => $vehicle->colour,
                    'FuelType' => $vehicle->fuelType?->name,
                    'BodyType' => $vehicle->bodyStyle?->name,
                    'Doors' => $vehicle->door_count,
                    'EngineSize' => $vehicle->engine_size_cc,
                    'Transmission' =>  $vehicle->transmission?->name,
                    'Description' => $vehicle->description,
                    'Options' => $vehicle->specs
                        ->pluck('description')
                        ->implode(','),

                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'Feed_Id',
            'Vehicle_Id',
            'Cap_Id',
            'Make',
            'Model',
            'Variant',
            'Year',
            'Price',
            'FullRegistration',
            'Mileage',
            'PictureRefs',
            'Vehicle_URL',
            'Colour',
            'FuelType',
            'BodyType',
            'Doors',
            'EngineSize',
            'Transmission',
            'Description',
            'Options',
        ];
    }
}
