<?php

namespace Tests\Feature;

use App\Facades\Settings;
use App\IntegrationRepository;
use App\Tier;
use App\TierHelper;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TierHelperTest extends TestCase
{
    public function testIsAllowed()
    {
        $this->assertTrue(TierHelper::isAllowed('lite', 'lite'));
        $this->assertFalse(TierHelper::isAllowed('lite', 'standard'));
        $this->assertFalse(TierHelper::isAllowed('lite', 'pro'));
        $this->assertFalse(TierHelper::isAllowed('lite', 'enterprise'));

        $this->assertTrue(TierHelper::isAllowed('standard', 'lite'));
        $this->assertTrue(TierHelper::isAllowed('standard', 'standard'));
        $this->assertFalse(TierHelper::isAllowed('standard', 'pro'));
        $this->assertFalse(TierHelper::isAllowed('standard', 'enterprise'));

        $this->assertTrue(TierHelper::isAllowed('pro', 'lite'));
        $this->assertTrue(TierHelper::isAllowed('pro', 'standard'));
        $this->assertTrue(TierHelper::isAllowed('pro', 'pro'));
        $this->assertFalse(TierHelper::isAllowed('pro', 'enterprise'));

        $this->assertTrue(TierHelper::isAllowed('enterprise', 'lite'));
        $this->assertTrue(TierHelper::isAllowed('enterprise', 'standard'));
        $this->assertTrue(TierHelper::isAllowed('enterprise', 'pro'));
        $this->assertTrue(TierHelper::isAllowed('enterprise', 'enterprise'));
    }

    public function testIsUpgrade()
    {
        $this->assertFalse(TierHelper::isUpgrade('lite', 'lite'));
        $this->assertTrue(TierHelper::isUpgrade('lite', 'standard'));
        $this->assertTrue(TierHelper::isUpgrade('lite', 'pro'));
        $this->assertTrue(TierHelper::isUpgrade('lite', 'enterprise'));

        $this->assertFalse(TierHelper::isUpgrade('standard', 'lite'));
        $this->assertFalse(TierHelper::isUpgrade('standard', 'standard'));
        $this->assertTrue(TierHelper::isUpgrade('standard', 'pro'));
        $this->assertTrue(TierHelper::isUpgrade('standard', 'enterprise'));

        $this->assertFalse(TierHelper::isUpgrade('pro', 'lite'));
        $this->assertFalse(TierHelper::isUpgrade('pro', 'standard'));
        $this->assertFalse(TierHelper::isUpgrade('pro', 'pro'));
        $this->assertTrue(TierHelper::isUpgrade('pro', 'enterprise'));

        $this->assertFalse(TierHelper::isUpgrade('enterprise', 'lite'));
        $this->assertFalse(TierHelper::isUpgrade('enterprise', 'standard'));
        $this->assertFalse(TierHelper::isUpgrade('enterprise', 'pro'));
        $this->assertFalse(TierHelper::isUpgrade('enterprise', 'enterprise'));
    }

    public function testAllowedFeatures()
    {
        $lite = TierHelper::allowedFeatures(Tier::LITE);
        $standard = TierHelper::allowedFeatures(Tier::STANDARD);
        $pro = TierHelper::allowedFeatures(Tier::PRO);
        $enterprise = TierHelper::allowedFeatures(Tier::ENTERPRISE);

        $this->assertEmpty($lite);
        $this->assertTrue(in_array('plugin', $standard));
        $this->assertFalse(in_array('custom', $standard));
        $this->assertTrue(in_array('add-on', $pro));
        $this->assertFalse(in_array('theme', $pro));
        $this->assertTrue(in_array('plugin', $pro));
        $this->assertTrue(in_array('plugin', $enterprise));
        $this->assertTrue(in_array('add-on', $enterprise));
        $this->assertTrue(in_array('theme', $enterprise));
        $this->assertTrue(in_array('custom', $enterprise));
    }

    public function testDisallowedFeatures()
    {
        $lite = TierHelper::disallowedFeatures(Tier::LITE);
        $standard = TierHelper::disallowedFeatures(Tier::STANDARD);
        $pro = TierHelper::disallowedFeatures(Tier::PRO);
        $enterprise = TierHelper::disallowedFeatures(Tier::ENTERPRISE);

        $this->assertEmpty($enterprise);
        $this->assertTrue(in_array('theme', $pro));
        $this->assertTrue(in_array('custom', $pro));
        $this->assertTrue(in_array('add-on', $standard));
        $this->assertTrue(in_array('theme', $standard));
        $this->assertTrue(in_array('custom', $standard));
        $this->assertTrue(in_array('plugin', $lite));
        $this->assertTrue(in_array('add-on', $lite));
        $this->assertTrue(in_array('theme', $lite));
        $this->assertTrue(in_array('custom', $lite));
    }

    public function testUserLimit()
    {
        $lite = TierHelper::userLimit(Tier::LITE->value);
        $standard = TierHelper::userLimit(Tier::STANDARD->value);
        $pro = TierHelper::userLimit(Tier::PRO->value);
        $enterprise = TierHelper::userLimit(Tier::ENTERPRISE->value);

        $this->assertEquals(2, $lite);
        $this->assertEquals(5, $standard);
        $this->assertEquals(999, $pro);
        $this->assertEquals(999, $enterprise);
    }

    public function testTiersAbove()
    {
        $lite = TierHelper::tiersAbove(Tier::LITE->value);
        $standard = TierHelper::tiersAbove(Tier::STANDARD->value);
        $pro = TierHelper::tiersAbove(Tier::PRO->value);
        $enterprise = TierHelper::tiersAbove(Tier::ENTERPRISE->value);

        $this->assertTrue(in_array('pro', $lite));
        $this->assertCount(3, $lite);

        $this->assertTrue(in_array('pro', $standard));
        $this->assertCount(2, $standard);

        $this->assertTrue(in_array('enterprise', $pro));
        $this->assertCount(1, $pro);

        $this->assertCount(0, $enterprise);
    }
}
