<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Traits\HasPlaceholderImage;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;

class StockAndOfferList extends VehicleList
{
    use HasPlaceholderImage;

    public function mapVehicle(Vehicle $vehicle)
    {
        $vehicle->offer_data = $vehicle->is_offer
            ? VehicleOffer::query()->where('id', '=', $vehicle->id)->first()
            : null;

        $data = [
            'id' => $vehicle->id,
            'title' => $vehicle->title,
            'derivative' => $vehicle->derivative,
            'slug' => $vehicle->slug,
            'is_new' => $vehicle->is_new,
            'is_reserved' => $vehicle->is_reserved,
            'price' => $vehicle->price,
            'make' => [
                'name' => $vehicle->make->name ?? '',
                'slug' => $vehicle->make->slug ?? '',
                'logo' => Settings::get('automotive-vehicle-brand-on-filter-card')
                    ? $vehicle->make->logo ?? ''
                    : null
            ],
            'location' => $this->locationValue($vehicle),
            'deposit' => $vehicle->deposit,
            'dealership' => $this->getDealership($vehicle),
            'fuel_type' => $vehicle->fuelType?->name,
            'transmission' => $vehicle->transmission?->name,
            'odometer_mi' => $vehicle->odometer_mi,
            'odometer_km' => $vehicle->odometer_km,
            'manufacture_year' => $vehicle->manufacture_year,
            'registration_number' => $vehicle->registration_number,
            'model' => Settings::get('automotive-vehicle-model-on-filter-card')
                ? [
                    'name' => $vehicle->model->name ?? '',
                    'slug' => $vehicle->model->slug ?? '',
                ]
                : $vehicle->model?->name,
            'original_price' => $vehicle->original_price,
            'previous_price' => $vehicle->previous_price,
            'rrp_price' => $vehicle->rrp_price,
            'type' => $vehicle->type,
            'extra' => $this->getExtraData($vehicle),
            'seo' => $vehicle->seo,
            'age_identifier' => $vehicle->getAgeIdentifier(),
            'custom' => $vehicle->getAllCustom(),
            'monthly_price' => $vehicle->monthly_price,
            'labels' => Settings::get('vehicle-labels-enabled')
                ? $vehicle->labels->whereIn('type', ['', 'all', 'srp'])->values()
                : [],
            'engine_size_cc' => $vehicle->engine_size_cc,
            'is_vat_applicable' => $vehicle->is_vat_applicable,
            'coming_soon' => $vehicle->getComingSoonStatus(),
            'attention_grabber' => $vehicle->attention_grabber,
            'images_count' => $vehicle->image_count ?? 0,
            'features' => $vehicle->features->pluck('name'),
            'main_video_url' =>  $vehicle->main_video_url ?? '',
            'is_offer' => $vehicle->is_offer ? true : false,
            'colour' => $vehicle->colour,
            'trim' => $vehicle->trim,
        ];

        $data = array_merge($data, $this->getFinanceData($vehicle));

        if (Settings::get('autotrader-advert-performance')) {
            $data['autoTraderData'] =  [
                'price_point' => $vehicle->autoTraderData?->price_point,
            ];
        }
        if (Settings::get('vehicle-card-image-count') > 1) {
            $data['gallery'] = $this->getGallery($vehicle, Settings::get('vehicle-card-image-count'));
        } else {
            $data['thumbnail'] = [
                'small' => $this->getStockOfferPreviewImage($vehicle),
            ];
        }

        return $data;
    }

    protected function getFinanceData(Vehicle $vehicle): array
    {
        $finance_example = $this->getStockOfferFinanceExample($vehicle);

        return [
            'finance_example' => $finance_example,
            'monthly_cost_type' => $finance_example?->finance_type,
        ];
    }

    protected function getStockOfferFinanceExample(Vehicle $vehicle)
    {
        if ($vehicle->is_offer) {
            return $vehicle->offer_data?->cheapestFinance;
        }

        return $this->getFinanceExample($vehicle);
    }

    protected function getStockOfferPreviewImage(Vehicle $vehicle): string
    {
        $preview_image = $vehicle->is_offer
            ? $vehicle->offer_data->getPreviewImage($this->mediumImageDimensionsOffer())
            : $vehicle->getPreviewImage($this->mediumImageDimensionsStock());

        return $preview_image ?? $this->getPlaceholder($vehicle, $this->mediumImageDimensionsStock());
    }

    protected function mediumImageDimensionsOffer(): string
    {
        return 'medium';
    }

    protected function mediumImageDimensionsStock(): string
    {
        return 'tile';
    }
}
