<?php

namespace App\Http\Resources;

use App\Facades\Feature;
use App\Facades\Settings;
use App\Modules\PlaceholderImages\ImaginStudio;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Mtc\MercuryDataModels\Leasing\KeyloopLeaseFinanceService;

class LeaseVehicleList extends JsonResource
{
    /**
     * Transform the resource collection into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array
     */
    #[\Override]
    public function toArray($request)
    {
        if (Settings::get('lease-vehicle-labels-enabled')) {
            $this->resource->load([
                'labels'
            ]);
        }

        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource
                ->through(fn ($vehicle) => $this->mapVehicle($vehicle));
        } else {
            $this->resource = $this->resource
                ->map(fn($vehicle) => $this->mapVehicle($vehicle));
        }

        $data = $this->resource->toArray();
        $data['finance_settings'] = [
            'contract_terms' => config('leasing.keyloop.contract_terms'),
            'deposit_quantity_months' => config('leasing.keyloop.deposit_quantity_months'),
            'annual_mileages' => config('leasing.keyloop.annual_mileages'),
        ];

        return $data;
    }

    public function mapVehicle(KeyloopLeaseVehicleVariant $vehicle): array
    {
        $data = [
            'id' => $vehicle->id,
            'slug' => $vehicle->slug,
            'external_variant_id' => $vehicle->external_variant_id,
            'title' => $vehicle->title,
            'name' => $vehicle->name,
            'derivative' => $vehicle->derivative,
            'price' => $vehicle->price,
            'make' => [
                'name' => $vehicle->make->name ?? '',
                'slug' => $vehicle->make->slug ?? '',
                'logo' => Settings::get('automotive-vehicle-brand-on-filter-card')
                    ? $vehicle->make->logo ?? ''
                    : null
            ],
            'fuel_type' => $vehicle->fuelType?->name,
            'transmission' => $vehicle->transmission?->name,
            'body_type' => $vehicle->bodyStyle?->name,
            'manufacture_year' => $vehicle->manufacture_year,
            'model' => Settings::get('automotive-vehicle-model-on-filter-card')
                ? [
                    'name' => $vehicle->model->name ?? '',
                    'slug' => $vehicle->model->slug ?? '',
                ]
                : $vehicle->model?->name,
            'type' => $vehicle->type,
            'seo' => $vehicle->seo,
            'monthly_from_price_ex_vat' => $vehicle->cheapest_monthly_price_ex_vat,
            'monthly_from_price_inc_vat' => $vehicle->cheapest_monthly_price_inc_vat,
            'price_ex_vat_ex_vrt' => $vehicle->price_ex_vat_ex_vrt,
            'price_inc_vat_ex_vrt' => $vehicle->price_inc_vat_ex_vrt,
            'engine_size_cc' => $vehicle->engine_size_cc,
            'is_vat_applicable' => $vehicle->is_vat_applicable,
            'attention_grabber' => $vehicle->attention_grabber,
            'labels' => Settings::get('lease-vehicle-labels-enabled')
                ? $vehicle->labels->whereIn('type', ['', 'all', 'srp'])->values()
                : [],
            'rrp' => $this->getRrp($vehicle),
            'colours' => $this->coloursForDisplay($vehicle),
        ];

        $data['imagin_studio_base_url'] = Feature::isEnabled('imagin-studio-placeholders')
            ? new ImaginStudio()->getBaseUrl()
            : null;

        $data['finance_options'] = new KeyloopLeaseFinanceService()->getVariantFinance($vehicle);

        return $data;
    }

    protected function getRrp(KeyloopLeaseVehicleVariant $vehicle)
    {
        return $vehicle->price_inc_vat_inc_vrt;
    }

    protected function coloursForDisplay(KeyloopLeaseVehicleVariant $vehicle)
    {
        $colours = $vehicle->coloursRelation?->map(fn($item) => [
            'id' => $item->id,
            'name' => $item->name,
            'colour_code' => $item->colour_code,

        ])->toArray();

        shuffle($colours);
        return $colours;
    }
}
